/*
 * Copyright(c) 2007 Atheros Corporation. All rights reserved.
 *
 * Derived from Intel e1000 driver
 * Copyright(c) 1999 - 2005 Intel Corporation. All rights reserved.
 *
 * This program is free software; you can redistribute it and/or modify it
 * under the terms of the GNU General Public License as published by the Free
 * Software Foundation; either version 2 of the License, or (at your option)
 * any later version.
 *
 * This program is distributed in the hope that it will be useful, but WITHOUT
 * ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or
 * FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public License for
 * more details.
 *
 * You should have received a copy of the GNU General Public License along with
 * this program; if not, write to the Free Software Foundation, Inc., 59
 * Temple Place - Suite 330, Boston, MA  02111-1307, USA.
 *
 * There are a lot of defines in here that are unused and/or have cryptic
 * names.  Please leave them alone, as they're the closest thing we have
 * to a spec from Atheros at present. *ahem* -- CHS
 */


#include "at.h"

#ifdef CONFIG_AT_MQ
#include <linux/cpu.h>
#include <linux/smp.h>
#endif
#ifdef USE_REBOOT_NOTIFIER
#include <linux/reboot.h>
#endif

char at_driver_name[] = "ATL1";
char at_driver_string[] = "Atheros(R) L1 Ethernet Network Driver";
#define DRV_VERSION "1.2.40.3"
char at_driver_version[] = DRV_VERSION;
char at_copyright[] = "Copyright (c) 2007 Atheros Corporation.";

/* 
 * at_pci_tbl - PCI Device ID Table
 *
 * Wildcard entries (PCI_ANY_ID) should come last
 * Last entry must be all 0s
 *
 * { Vendor ID, Device ID, SubVendor ID, SubDevice ID,
 *   Class, Class Mask, private data (not used) }
 */
static struct pci_device_id at_pci_tbl[] = {
    ATHEROS_ETHERNET_DEVICE(0x1048),
    /* required last entry */
    {0,}
};

MODULE_DEVICE_TABLE(pci, at_pci_tbl);

int at_up(struct at_adapter *adapter);
void at_down(struct at_adapter *adapter);
int at_reset(struct at_adapter *adapter);
s32 at_setup_ring_resources(struct at_adapter *adapter);
void at_free_ring_resources(struct at_adapter *adapter);
void at_reinit_locked(struct at_adapter *adapter);


/* Local Function Prototypes */
static int at_init_module(void);
static void at_exit_module(void);
static int at_probe(struct pci_dev *pdev, const struct pci_device_id *ent);
static int at_alloc_queues(struct at_adapter *adapter);

#ifdef CONFIG_AT_NAPI
static int at_clean(struct net_device *poll_dev, int *budget);
#endif

static void __devexit at_remove(struct pci_dev *pdev);
static int at_sw_init(struct at_adapter *adapter);
static int at_open(struct net_device *netdev);
static int at_close(struct net_device *netdev);
static int at_xmit_frame(struct sk_buff *skb, struct net_device *netdev);
static struct net_device_stats * at_get_stats(struct net_device *netdev);
static int at_tso(struct at_adapter* adapter, struct sk_buff *skb, TpdDescr* pTpd);
static void at_tx_map(struct at_adapter *adapter, struct sk_buff *skb, TpdDescr* pTpd);
static void at_tx_queue(struct at_adapter *adapter, u16 count, TpdDescr* pTpd);
static int at_tx_csum(struct at_adapter *adapter, struct sk_buff *skb, TpdDescr* pTpd);
static int at_change_mtu(struct net_device *netdev, int new_mtu);
static void at_set_multi(struct net_device *netdev);
static int at_set_mac(struct net_device *netdev, void *p);
static int at_ioctl(struct net_device *netdev, struct ifreq *ifr, int cmd);
static void at_tx_timeout(struct net_device *dev);
static irqreturn_t at_intr(int irq, void *data);
static boolean_t at_clean_tx_irq(struct at_adapter* adapter);


static void at_watchdog(unsigned long data);
static void at_phy_config(unsigned long data);
static void at_reset_task(struct work_struct *work);
static void at_link_chg_task(struct work_struct *work);
static void at_check_for_link(struct at_adapter* adapter);
void at_set_ethtool_ops(struct net_device *netdev);
#ifdef ETHTOOL_OPS_COMPAT
extern int ethtool_ioctl(struct ifreq *ifr);
#endif
static int at_check_link(struct at_adapter* adapter);
void init_ring_ptrs(struct at_adapter *adapter);
static s32 at_configure(struct at_adapter *adapter);
#ifdef CONFIG_AT_NAPI
static void at_clean_rx_irq(struct at_adapter *adapter, int *work_done, int work_to_do);
#else
static void at_clean_rx_irq(struct at_adapter *adapter);
#endif

static void at_clean_tx_ring(struct at_adapter *adapter);
static void at_clean_rx_ring(struct at_adapter *adapter);
static u16 at_alloc_rx_buffers(struct at_adapter *adapter);
static void at_inc_smb(struct at_adapter * adapter);
    
#ifdef SIOCGMIIPHY
static int at_mii_ioctl(struct net_device *netdev, struct ifreq *ifr, int cmd);
#endif

#ifdef NETIF_F_HW_VLAN_TX
static void at_vlan_rx_register(struct net_device *netdev, struct vlan_group *grp);
static void at_restore_vlan(struct at_adapter *adapter);
#endif

static int at_suspend(struct pci_dev *pdev, pm_message_t state);
#ifdef CONFIG_PM
static int at_resume(struct pci_dev *pdev);
#endif

#ifndef USE_REBOOT_NOTIFIER
static void at_shutdown(struct pci_dev *pdev);
#else
static int at_notify_reboot(struct notifier_block *nb, unsigned long event, void *p);

struct notifier_block at_notifier_reboot = {
    .notifier_call  = at_notify_reboot,
    .next       = NULL,
    .priority   = 0
};
#endif

#ifdef CONFIG_NET_POLL_CONTROLLER
/* for netdump / net console */
static void at_netpoll (struct net_device *netdev);
#endif


/* Exported from other modules */

extern void at_check_options(struct at_adapter *adapter);


#ifdef CONFIG_AT_PCI_ERS
static pci_ers_result_t at_io_error_detected(struct pci_dev *pdev,
                     pci_channel_state_t state);
static pci_ers_result_t at_io_slot_reset(struct pci_dev *pdev);
static void at_io_resume(struct pci_dev *pdev);

static struct pci_error_handlers at_err_handler = {
    .error_detected = at_io_error_detected,
    .slot_reset = at_io_slot_reset,
    .resume = at_io_resume,
};
#endif

static struct pci_driver at_driver = {
    .name     = at_driver_name,
    .id_table = at_pci_tbl,
    .probe    = at_probe,
    .remove   = __devexit_p(at_remove),
    /* Power Managment Hooks */
#ifdef CONFIG_PM
    .suspend  = at_suspend,
    .resume   = at_resume,
#endif
#ifndef USE_REBOOT_NOTIFIER
    .shutdown = at_shutdown,
#endif
#ifdef CONFIG_AT_PCI_ERS
    .err_handler = &at_err_handler
#endif

};

MODULE_AUTHOR("Atheros Corporation, <xiong.huang@atheros.com>");
MODULE_DESCRIPTION("Atheros 1000M Ethernet Network Driver");
MODULE_LICENSE("GPL");
MODULE_VERSION(DRV_VERSION);

/**
 * at_init_module - Driver Registration Routine
 *
 * at_init_module is the first routine called when the driver is
 * loaded. All it does is register with the PCI subsystem.
 **/

static int __init
at_init_module(void)
{
    int ret;
    printk(KERN_INFO "%s - version %s\n",
           at_driver_string, at_driver_version);

    printk(KERN_INFO "%s\n", at_copyright);

    ret = pci_register_driver(&at_driver);
#ifdef USE_REBOOT_NOTIFIER
    if (ret >= 0) {
        register_reboot_notifier(&at_notifier_reboot);
    }
#endif
    
    return ret;
}

module_init(at_init_module);

/**
 * at_exit_module - Driver Exit Cleanup Routine
 *
 * at_exit_module is called just before the driver is removed
 * from memory.
 **/

static void __exit
at_exit_module(void)
{
#ifdef USE_REBOOT_NOTIFIER
    unregister_reboot_notifier(&at_notifier_reboot);
#endif
    pci_unregister_driver(&at_driver);
    
}

module_exit(at_exit_module);



static int at_request_irq(struct at_adapter *adapter)
{
    struct net_device *netdev = adapter->netdev;
    int flags, err = 0;

    flags = IRQF_SHARED;
#ifdef CONFIG_PCI_MSI
    adapter->have_msi = TRUE;
    if ((err = pci_enable_msi(adapter->pdev))) {
        AT_DBG("Unable to allocate MSI interrupt Error: %d\n", err);
        adapter->have_msi = FALSE;
    }
    
    if (adapter->have_msi) {
        flags &= ~IRQF_SHARED;
    }
#endif

    if ((err = request_irq(adapter->pdev->irq, &at_intr, flags,
                           netdev->name, netdev))) {
        AT_DBG("Unable to allocate interrupt Error: %d\n", err);                            
    }
    
    AT_DBG("at_request_irq OK\n");

    return err;
}


static void at_free_irq(struct at_adapter *adapter)
{
    struct net_device *netdev = adapter->netdev;

    free_irq(adapter->pdev->irq, netdev);

#ifdef CONFIG_PCI_MSI
    if (adapter->have_msi)
        pci_disable_msi(adapter->pdev);
#endif
}


static void at_setup_pcicmd(struct pci_dev* pdev)
{
    u16 cmd;

    pci_read_config_word(pdev, PCI_COMMAND, &cmd);
    
    if (cmd & PCI_COMMAND_INTX_DISABLE)
        cmd &= ~PCI_COMMAND_INTX_DISABLE;
    if (cmd & PCI_COMMAND_IO)
        cmd &= ~PCI_COMMAND_IO;
    if (0 == (cmd & PCI_COMMAND_MEMORY))
        cmd |= PCI_COMMAND_MEMORY;
    if (0 == (cmd & PCI_COMMAND_MASTER))
        cmd |= PCI_COMMAND_MASTER;
    pci_write_config_word(pdev, PCI_COMMAND, cmd);
    
    /* 
     * some motherboards BIOS(PXE/EFI) driver may set PME
     * while they transfer control to OS (Windows/Linux)
     * so we should clear this bit before NIC work normally
     */
    pci_write_config_dword(pdev, REG_PM_CTRLSTAT, 0); 
    msec_delay(1);          
}


/**
 * at_probe - Device Initialization Routine
 * @pdev: PCI device information struct
 * @ent: entry in at_pci_tbl
 *
 * Returns 0 on success, negative on failure
 *
 * at_probe initializes an adapter identified by a pci_dev structure.
 * The OS initialization, configuring of the adapter private structure,
 * and a hardware reset occur.
 **/

static int __devinit
at_probe(struct pci_dev *pdev,
            const struct pci_device_id *ent)
{
    struct net_device *netdev;
    struct at_adapter *adapter;
    static int cards_found = 0;
    unsigned long mmio_start;
    int mmio_len;
    boolean_t pci_using_64 = TRUE;
    int err;
#ifdef CONFIG_AT_NAPI
    int i;
#endif

    DEBUGFUNC("at_probe !");

    if((err = pci_enable_device(pdev)))
        return err;

    if (!(err = pci_set_dma_mask(pdev, DMA_64BIT_MASK)) &&
        !(err = pci_set_consistent_dma_mask(pdev, DMA_64BIT_MASK))) {
        pci_using_64 = TRUE; 
    } else {
        if ((err = pci_set_dma_mask(pdev, DMA_32BIT_MASK)) &&
            (err = pci_set_consistent_dma_mask(pdev, DMA_32BIT_MASK))) {
            AT_ERR("No usable DMA configuration, aborting\n");
            goto err_dma;
        }
        pci_using_64 = FALSE;
    }       
        
   
    // Mark all PCI regions associated with PCI device 
    // pdev as being reserved by owner at_driver_name
    if((err = pci_request_regions(pdev, at_driver_name)))
       goto err_pci_reg;

    // Enables bus-mastering on the device and calls 
    // pcibios_set_master to do the needed arch specific settings
    pci_set_master(pdev);

    err = -ENOMEM;
    netdev = alloc_etherdev(sizeof(struct at_adapter));
    if(!netdev)
        goto err_alloc_etherdev;
    
    SET_MODULE_OWNER(netdev);
    SET_NETDEV_DEV(netdev, &pdev->dev);

    pci_set_drvdata(pdev, netdev);
    adapter = netdev_priv(netdev);
    adapter->netdev = netdev;
    adapter->pdev = pdev;
    adapter->hw.back = adapter;

    mmio_start = pci_resource_start(pdev, BAR_0);
    mmio_len = pci_resource_len(pdev, BAR_0);

    AT_DBG("base memory = %lx memory length = %x \n", 
        mmio_start, mmio_len);
    adapter->hw.hw_addr = ioremap(mmio_start, mmio_len);
    if(!adapter->hw.hw_addr) {
        err = -EIO;
        goto err_ioremap;
    }
    
    at_setup_pcicmd(pdev);

    netdev->open = &at_open;
    netdev->stop = &at_close;
    netdev->hard_start_xmit = &at_xmit_frame;
    netdev->get_stats = &at_get_stats;
    netdev->set_multicast_list = &at_set_multi;
    netdev->set_mac_address = &at_set_mac;
    netdev->change_mtu = &at_change_mtu;
    netdev->do_ioctl = &at_ioctl;
    at_set_ethtool_ops(netdev);
    
#ifdef HAVE_TX_TIMEOUT
    netdev->tx_timeout = &at_tx_timeout;
    netdev->watchdog_timeo = 5 * HZ;
#endif
#ifdef CONFIG_AT_NAPI
    netdev->poll = &at_clean;
    netdev->weight = 64;
#endif
#ifdef NETIF_F_HW_VLAN_TX
    netdev->vlan_rx_register = at_vlan_rx_register;
#endif
#ifdef CONFIG_NET_POLL_CONTROLLER
    netdev->poll_controller = at_netpoll;
#endif
    strncpy(netdev->name, pci_name(pdev), sizeof(netdev->name) - 1);
    
    netdev->mem_start = mmio_start;
    netdev->mem_end = mmio_start + mmio_len;
    adapter->bd_number = cards_found;
    adapter->pci_using_64 = pci_using_64;

    /* setup the private structure */

    if((err = at_sw_init(adapter)))
        goto err_sw_init;

    err = -EIO;

#ifdef MAX_SKB_FRAGS
#ifdef NETIF_F_HW_VLAN_TX
        netdev->features = NETIF_F_SG |
                   NETIF_F_HW_CSUM |
                   NETIF_F_HW_VLAN_TX |
                   NETIF_F_HW_VLAN_RX ;
#else
        netdev->features = NETIF_F_SG | NETIF_F_HW_CSUM;
#endif

#ifdef NETIF_F_TSO
        netdev->features |= NETIF_F_TSO;
#endif//NETIF_F_TSO
#endif//MAX_SKB_FRAGS

#ifdef NETIF_F_LLTX
    netdev->features |= NETIF_F_LLTX;
#endif

    if(pci_using_64) {
        netdev->features |= NETIF_F_HIGHDMA;
        AT_DBG("pci using 64bit address\n");
    }

    /* Init GPHY as early as possible due to power saving issue  */
    at_phy_init(&adapter->hw);


    /* reset the controller to 
     * put the device in a known good starting state */
    
    if (at_reset_hw(&adapter->hw)) {
        err = -EIO;
        goto err_reset;
    }

    /* copy the MAC address out of the EEPROM */

    at_read_mac_addr(&adapter->hw);
    memcpy(netdev->dev_addr, adapter->hw.mac_addr, netdev->addr_len);
#ifdef ETHTOOL_GPERMADDR
    memcpy(netdev->perm_addr, adapter->hw.mac_addr, netdev->addr_len);

    if (!is_valid_ether_addr(netdev->perm_addr)) {
#else
    if (!is_valid_ether_addr(netdev->dev_addr)) {
#endif
        AT_DBG("Invalid MAC Address\n");
        err = -EIO;
        goto err_eeprom;
    }
    AT_DBG("mac address : %02x-%02x-%02x-%02x-%02x-%02x\n",
        adapter->hw.mac_addr[0],
        adapter->hw.mac_addr[1],
        adapter->hw.mac_addr[2],
        adapter->hw.mac_addr[3],
        adapter->hw.mac_addr[4],
        adapter->hw.mac_addr[5] );

    at_check_options(adapter);
    
    init_timer(&adapter->watchdog_timer);
    adapter->watchdog_timer.function = &at_watchdog;
    adapter->watchdog_timer.data = (unsigned long) adapter;
    
    init_timer(&adapter->phy_config_timer);
    adapter->phy_config_timer.function = &at_phy_config;
    adapter->phy_config_timer.data = (unsigned long) adapter;
    
    INIT_WORK(&adapter->reset_task, at_reset_task);
    INIT_WORK(&adapter->link_chg_task, at_link_chg_task);


    strcpy(netdev->name, "eth%d"); // ??
    if((err = register_netdev(netdev)))
        goto err_register;

    /* assume we have no link for now */
    netif_carrier_off(netdev);
    netif_stop_queue(netdev);
    
    
    cards_found++;

    return 0;

//err_init_hw:
err_reset:
err_register:   
err_sw_init:
err_eeprom:
    
#ifdef CONFIG_AT_NAPI
    if (adapter->polling_netdev) {
        for (i = 0; i < adapter->num_rx_queues; i++)
            dev_put(&adapter->polling_netdev[i]);   
        kfree(adapter->polling_netdev);
    }
 #endif 
 
    iounmap(adapter->hw.hw_addr);
err_ioremap:
    free_netdev(netdev);
err_alloc_etherdev:
    pci_release_regions(pdev);
err_pci_reg:
err_dma:
    pci_disable_device(pdev);
    return err;
}

/**
 * at_remove - Device Removal Routine
 * @pdev: PCI device information struct
 *
 * at_remove is called by the PCI subsystem to alert the driver
 * that it should release a PCI device.  The could be caused by a
 * Hot-Plug event, or because the driver is going to be removed from
 * memory.
 **/

static void __devexit
at_remove(struct pci_dev *pdev)
{
    struct net_device *netdev = pci_get_drvdata(pdev);
    struct at_adapter *adapter = netdev_priv(netdev);
#ifdef CONFIG_AT_NAPI
    int i;
#endif
    
    DEBUGFUNC("at_remove");


    /* flush_scheduled work may reschedule our watchdog task, so
     * explicitly disable watchdog tasks from being rescheduled  */
    set_bit(__AT_DOWN, &adapter->flags);

    del_timer_sync(&adapter->watchdog_timer);
    del_timer_sync(&adapter->phy_config_timer);

    flush_scheduled_work();

    unregister_netdev(netdev);

#ifdef CONFIG_AT_NAPI
    for (i = 0; i < adapter->num_rx_queues; i++)
        dev_put(&adapter->polling_netdev[i]);

    kfree(adapter->polling_netdev);

#endif

	at_force_ps(&adapter->hw);

    iounmap(adapter->hw.hw_addr);
    pci_release_regions(pdev);
 
    free_netdev(netdev);

    pci_disable_device(pdev);
}

#ifdef USE_REBOOT_NOTIFIER
/* only want to do this for 2.4 kernels? */
static int
at_notify_reboot(struct notifier_block *nb, unsigned long event, void *p)
{
    struct pci_dev *pdev = NULL;

    DEBUGFUNC("at_notify_reboot !");

    switch(event) {
    case SYS_DOWN:
    case SYS_HALT:
    case SYS_POWER_OFF:
        while((pdev = pci_find_device(PCI_ANY_ID, PCI_ANY_ID, pdev))) {
            if(pci_dev_driver(pdev) == &at_driver)
                at_suspend(pdev, PMSG_SUSPEND);
        }
    }
    return NOTIFY_DONE;
    
}
#endif

#ifndef USE_REBOOT_NOTIFIER
static void at_shutdown(struct pci_dev *pdev)
{
    at_suspend(pdev, PMSG_SUSPEND);
}
#endif

#ifdef CONFIG_NET_POLL_CONTROLLER
/*
 * Polling 'interrupt' - used by things like netconsole to send skbs
 * without having to re-enable interrupts. It's not called while
 * the interrupt routine is executing.
 */
static void
at_netpoll(struct net_device *netdev)
{
    struct at_adapter *adapter = netdev_priv(netdev);

    disable_irq(adapter->pdev->irq);
    at_intr(adapter->pdev->irq, netdev);
    at_clean_tx_irq(adapter);
#ifndef CONFIG_AT_NAPI
    at_clean_rx_irq(adapter);
#endif
    enable_irq(adapter->pdev->irq);
}
#endif


static int
at_suspend(struct pci_dev *pdev, pm_message_t state)
{
#define AT_SUSPEND_LINK_TIMEOUT 28
    struct net_device *netdev = pci_get_drvdata(pdev);
    struct at_adapter *adapter = netdev_priv(netdev);
    struct at_hw * hw = &adapter->hw;
    //u16 speed, duplex;
    u32 ctrl = 0;
    u32 mac_ctrl_data = 0;
    u32 wol_ctrl_data = 0;
    u16 mii_advertise_data = 0;
    u16 mii_bmsr_data = 0;
    u16 mii_intr_status_data = 0;
    u32 wufc = adapter->wol;
    u32 i;
#ifdef CONFIG_PM
    int retval = 0;
#endif

    DEBUGFUNC("at_suspend !"); 

    if (netif_running(netdev)) {
        WARN_ON(test_bit(__AT_RESETTING, &adapter->flags));
        at_down(adapter);
    }

    netif_device_detach(netdev);

#ifdef CONFIG_PM
    retval = pci_save_state(pdev);
    if (retval)
        return retval;
#endif

    if (wufc) {
        /* get link status */  
        at_read_phy_reg(hw, MII_BMSR, (u16 *)&mii_bmsr_data);
        at_read_phy_reg(hw, MII_BMSR, (u16 *)&mii_bmsr_data);
        
        mii_advertise_data = hw->mii_autoneg_adv_reg & MII_AR_SPEED_MASK;
        mii_advertise_data |= MII_AR_10T_HD_CAPS; 
        printk(KERN_DEBUG "THe mii advertise data is %x\n", mii_advertise_data);

        if ((at_write_phy_reg(hw, MII_ADVERTISE, mii_advertise_data) != 0) || 
            (at_phy_commit(hw)) != 0) { 
            printk(KERN_DEBUG "set phy register failed\n");
        }
                 
        hw->phy_configured = FALSE; /* re-init PHY when resume */
          
        /* turn on magic packet wol */
        if (wufc & AT_WUFC_MAG) {
            wol_ctrl_data |= WOL_MAGIC_EN | WOL_MAGIC_PME_EN;
        }
        if (wufc & AT_WUFC_LNKC) {
            /* if orignal link status is link, just wait for retrive link */ 
            if (mii_bmsr_data & BMSR_LSTATUS) {                 
                
                for (i=0; i < AT_SUSPEND_LINK_TIMEOUT; i++) {
                    msec_delay(100);
                    at_read_phy_reg(hw, MII_BMSR, (u16 *)&mii_bmsr_data); 
                    if (mii_bmsr_data & BMSR_LSTATUS) {
                        break;
                    }
                }
                
                if (0 == (mii_bmsr_data & BMSR_LSTATUS)) {
                    printk(KERN_DEBUG "%s: Link may change when suspend\n",
                        at_driver_name);
                }
            }
            wol_ctrl_data |=  WOL_LINK_CHG_EN | WOL_LINK_CHG_PME_EN; 
            /* only link up can wake up */
            if (at_write_phy_reg(hw, 18, 0x400) != 0) {
                printk(KERN_DEBUG "%s: read write phy register failed.\n",
                        at_driver_name);
                goto wol_dis;
            }
        }
        /* clear phy interrupt */
        at_read_phy_reg(hw, 19, &mii_intr_status_data); 
        /* Config MAC Ctrl register */
        mac_ctrl_data = MAC_CTRL_RX_EN;
        /* set to 10/100M halt duplex */
        mac_ctrl_data |= MAC_CTRL_SPEED_10_100 << MAC_CTRL_SPEED_SHIFT;
        mac_ctrl_data |= (((u32)adapter->hw.preamble_len & MAC_CTRL_PRMLEN_MASK) 
                        << MAC_CTRL_PRMLEN_SHIFT);
        if (adapter->vlgrp) {
            mac_ctrl_data |= MAC_CTRL_RMV_VLAN;
        }
        if (wufc & AT_WUFC_MAG) {
            /* magic packet maybe Broadcast&multicast&Unicast frame */
            mac_ctrl_data |= MAC_CTRL_BC_EN;
        }
        AT_DBG("%s: suspend MAC=0x%x\n", at_driver_name, mac_ctrl_data);
            
        AT_WRITE_REG(hw, REG_WOL_CTRL, wol_ctrl_data);
        AT_WRITE_REG(hw, REG_MAC_CTRL, mac_ctrl_data);
        /* pcie patch */
        ctrl = AT_READ_REG(hw, REG_PCIE_PHYMISC);
        ctrl |= PCIE_PHYMISC_FORCE_RCV_DET;
        AT_WRITE_REG(hw, REG_PCIE_PHYMISC, ctrl);
        pci_enable_wake(pdev, pci_choose_state(pdev, state), 1);
        goto suspend_exit; 
    }    
wol_dis:
    
    /* WOL disabled */
    AT_WRITE_REG(hw, REG_WOL_CTRL, 0);
    
    /* pcie patch */
    ctrl = AT_READ_REG(hw, REG_PCIE_PHYMISC);
    ctrl |= PCIE_PHYMISC_FORCE_RCV_DET;
    AT_WRITE_REG(hw, REG_PCIE_PHYMISC, ctrl);               
    
    at_force_ps(hw);
    hw->phy_configured = FALSE; /* re-init PHY when resume */
    
    pci_enable_wake(pdev, pci_choose_state(pdev, state), 0);

suspend_exit:   

    if (netif_running(netdev))
        at_free_irq(adapter);

    pci_disable_device(pdev);

    pci_set_power_state(pdev, pci_choose_state(pdev, state));

    return 0;
#undef AT_SUSPEND_LINK_TIMEOUT
}


#ifdef CONFIG_PM
static int
at_resume(struct pci_dev *pdev)
{
    struct net_device *netdev = pci_get_drvdata(pdev);
    struct at_adapter *adapter = netdev_priv(netdev);
    u32 err;

    DEBUGFUNC("at_resume !");

    pci_set_power_state(pdev, PCI_D0);
    pci_restore_state(pdev);
    
    if ((err = pci_enable_device(pdev))) {
        printk(KERN_ERR "ATL1e: Cannot enable PCI device from suspend\n");
        return err;
    }
    
    pci_set_master(pdev);

    AT_READ_REG(&adapter->hw, REG_WOL_CTRL); /* clear WOL status */
    
    pci_enable_wake(pdev, PCI_D3hot, 0);
    pci_enable_wake(pdev, PCI_D3cold, 0);
    
    AT_WRITE_REG(&adapter->hw, REG_WOL_CTRL, 0);
    
    if (netif_running(netdev) && (err = at_request_irq(adapter)))
    return err;

    at_reset_hw(&adapter->hw);
    adapter->cmb.cmb->int_stats = 0;

    if(netif_running(netdev))
        at_up(adapter);

    netif_device_attach(netdev);

    return 0;
}
#endif


#ifdef CONFIG_AT_PCI_ERS
/**
 * at_io_error_detected - called when PCI error is detected
 * @pdev: Pointer to PCI device
 * @state: The current pci connection state
 *
 * This function is called after a PCI bus error affecting
 * this device has been detected.
 */
static pci_ers_result_t at_io_error_detected(struct pci_dev *pdev, pci_channel_state_t state)
{
    struct net_device *netdev = pci_get_drvdata(pdev);
    struct at_adapter *adapter = netdev->priv;

    netif_device_detach(netdev);

    if (netif_running(netdev))
        at_down(adapter);
        
    pci_disable_device(pdev);

    /* Request a slot slot reset. */
    return PCI_ERS_RESULT_NEED_RESET;
}

/**
 * at_io_slot_reset - called after the pci bus has been reset.
 * @pdev: Pointer to PCI device
 *
 * Restart the card from scratch, as if from a cold-boot. Implementation
 * resembles the first-half of the e1000_resume routine.
 */
static pci_ers_result_t at_io_slot_reset(struct pci_dev *pdev)
{
    struct net_device *netdev = pci_get_drvdata(pdev);
    struct at_adapter *adapter = netdev->priv;

    if (pci_enable_device(pdev)) {
        printk(KERN_ERR "ATL1e: Cannot re-enable PCI device after reset.\n");
        return PCI_ERS_RESULT_DISCONNECT;
    }
    pci_set_master(pdev);

    pci_enable_wake(pdev, PCI_D3hot, 0);
    pci_enable_wake(pdev, PCI_D3cold, 0);

    at_reset_hw(&adapter->hw);

    return PCI_ERS_RESULT_RECOVERED;
}

/**
 * at_io_resume - called when traffic can start flowing again.
 * @pdev: Pointer to PCI device
 *
 * This callback is called when the error recovery driver tells us that
 * its OK to resume normal operation. Implementation resembles the
 * second-half of the at_resume routine.
 */
static void at_io_resume(struct pci_dev *pdev)
{
    struct net_device *netdev = pci_get_drvdata(pdev);
    struct at_adapter *adapter = netdev->priv;

    if (netif_running(netdev)) {
        if (at_up(adapter)) {
            printk("ATL1e: can't bring device back up after reset\n");
            return;
        }
    }

    netif_device_attach(netdev);
}
#endif /* CONFIG_AT_PCI_ERS */



/**
 * at_irq_enable - Enable default interrupt generation settings
 * @adapter: board private structure
 **/

inline void
at_irq_enable(struct at_adapter *adapter)
{
#if 0
    int c = atomic_read(&adapter->irq_sem);
    AT_DBG("irq_sem=%d\n", c);
#endif

    if (likely(atomic_dec_and_test(&adapter->irq_sem))) {
        AT_WRITE_REG(&adapter->hw, REG_ISR, 0);
        AT_WRITE_REG(&adapter->hw, REG_IMR, IMR_NORMAL_MASK);
        AT_WRITE_FLUSH(&adapter->hw);
    }
}

/**
 * at_irq_disable - Mask off interrupt generation on the NIC
 * @adapter: board private structure
 **/

inline void
at_irq_disable(struct at_adapter *adapter)
{
    atomic_inc(&adapter->irq_sem);
    AT_WRITE_REG(&adapter->hw, REG_IMR, 0);
    AT_WRITE_FLUSH(&adapter->hw);
    synchronize_irq(adapter->pdev->irq);
}
    
/**
 * at_sw_init - Initialize general software structures (struct at_adapter)
 * @adapter: board private structure to initialize
 *
 * at_sw_init initializes the Adapter private data structure.
 * Fields are initialized based on PCI device information and
 * OS network device settings (MTU size).
 **/

static int __devinit
at_sw_init(struct at_adapter *adapter)
{
    struct at_hw *hw = &adapter->hw;
    struct pci_dev *pdev = adapter->pdev;
#ifdef CONFIG_AT_NAPI
    int i;
#endif

    /* PCI config space info */

    hw->vendor_id = pdev->vendor;
    hw->device_id = pdev->device;
    hw->subsystem_vendor_id = pdev->subsystem_vendor;
    hw->subsystem_id = pdev->subsystem_device;

    pci_read_config_byte(pdev, PCI_REVISION_ID, &hw->revision_id);
    pci_read_config_word(pdev, PCI_COMMAND, &hw->pci_cmd_word);
    /* default is wake on magic */
    adapter->wol = AT_WUFC_MAG;

    adapter->ict = 50000;  // 100ms
    
    adapter->link_speed = SPEED_0;   // hardware init
    adapter->link_duplex = FULL_DUPLEX; //

    hw->phy_configured = FALSE;
    hw->preamble_len = 7;
    hw->ipgt = 0x60;
    hw->min_ifg = 0x50;
    hw->ipgr1 = 0x40;
    hw->ipgr2 = 0x60;
    hw->max_retry = 0xf;
    hw->lcol = 0x37;
    hw->jam_ipg = 7;
    hw->max_frame_size = adapter->netdev->mtu;
    adapter->rx_buffer_len = (hw->max_frame_size+ 
                                ENET_HEADER_SIZE + 
                                VLAN_SIZE +
                                ETHERNET_FCS_SIZE + 7)&~7;    
    
    hw->rfd_burst = 8;
    hw->rrd_burst = 8;
    hw->rfd_fetch_gap = 1;
    hw->rx_jumbo_th = adapter->rx_buffer_len / 8;
    hw->rx_jumbo_lkah = 1;
    hw->rrd_ret_timer = 4; //8us
       
    hw->tpd_burst = 8;
    hw->tpd_fetch_th = 16; 
    hw->txf_burst = 0x200;
    hw->tx_jumbo_task_th = (hw->max_frame_size+
                                ENET_HEADER_SIZE + 
                                VLAN_SIZE +
                                ETHERNET_FCS_SIZE + 7)>>3;
    hw->tpd_fetch_gap = 1;
    
    hw->rcb_value = at_rcb_64;
    hw->dma_ord = at_dma_ord_out; //at_dma_ord_enh;
    hw->dmar_block = at_dma_req_1024;
    hw->dmaw_block = at_dma_req_1024;
    
    hw->cmb_rrd = 4; 
    hw->cmb_tpd = 4;
    hw->cmb_rx_timer = 2; //about 4us
    hw->cmb_tx_timer = 256; //about 512us
    hw->smb_timer = 100000 ; // about 200ms
    
    adapter->num_rx_queues = 1;

    if (at_alloc_queues(adapter)) {
        AT_ERR("Unable to allocate memory for queues\n");
        return -ENOMEM;
    }

#ifdef CONFIG_AT_NAPI
    for (i = 0; i < adapter->num_rx_queues; i++) {
        adapter->polling_netdev[i].priv = adapter;
        adapter->polling_netdev[i].poll = &at_clean;
        adapter->polling_netdev[i].weight = 64;
        dev_hold(&adapter->polling_netdev[i]);
        set_bit(__LINK_STATE_START, &adapter->polling_netdev[i].state);
    }
#endif


    atomic_set(&adapter->irq_sem, 1);
    spin_lock_init(&adapter->stats_lock);
    spin_lock_init(&adapter->tx_lock);
    spin_lock_init(&adapter->mb_lock);
    
    set_bit(__AT_DOWN, &adapter->flags);
    
    return 0;
}


#ifdef CONFIG_AT_NAPI
 /**
  * at_clean - NAPI Rx polling callback
  * @adapter: board private structure
  **/
 
static int
at_clean(struct net_device *poll_dev, int *budget)
{
    struct at_adapter *adapter;
    int work_to_do = min(*budget, poll_dev->quota);
    int i = 0, work_done = 0;
    boolean_t tx_cleaned = FALSE;
 
    /* Must NOT use netdev_priv macro here. */
    adapter = poll_dev->priv;
 
    /* Keep link state information with original netdev */
    if (!netif_carrier_ok(adapter->netdev))
        goto quit_polling;
 
    while (poll_dev != &adapter->polling_netdev[i]) {
        i++;
        if (unlikely(i == adapter->num_rx_queues))
            BUG();
    }
 
    /* at_clean is called per-cpu.  This lock protects
     * tx_ring[0] from being cleaned by multiple cpus
     * simultaneously.  A failure obtaining the lock means
     * tx_ring[0] is currently being cleaned anyway. */

    tx_cleaned = TRUE;

    at_clean_rx_irq(adapter, &work_done, work_to_do);
         
    *budget -= work_done;
    poll_dev->quota -= work_done;
 
    /* If no Tx and not enough Rx work done, exit the polling mode */
    if ((tx_cleaned && (work_done < work_to_do)) || !netif_running(poll_dev)) {
quit_polling:
        netif_rx_complete(poll_dev);
        if (test_bit(__AT_DOWN, &adapter->flags))
            atomic_dec(&adapter->irq_sem);
        else
            at_irq_enable(adapter);

        return 0;
    }
    return 1;
}
 
#endif
 
 
/**
  * at_alloc_queues - Allocate memory for all rings
  * @adapter: board private structure to initialize
  *
  * We allocate one ring per queue at run-time since we don't know the
  * number of queues at compile-time.  The polling_netdev array is
  * intended for Multiqueue, but should work fine with a single queue.
  **/
 
static int __devinit
at_alloc_queues(struct at_adapter *adapter)
{
#ifdef CONFIG_AT_NAPI
    int size;
#endif
 
#ifdef CONFIG_AT_NAPI
    size = sizeof(struct net_device) * adapter->num_rx_queues;
    adapter->polling_netdev = kmalloc(size, GFP_KERNEL);
    if (!adapter->polling_netdev) {
        return -ENOMEM;
    }
    memset(adapter->polling_netdev, 0, size);
#endif
 
    return AT_SUCCESS;
}

int
at_reset(struct at_adapter *adapter)
{
    int ret;
    
    if (AT_SUCCESS != (ret = at_reset_hw(&adapter->hw)))
        return ret;
        
    return at_init_hw(&adapter->hw);
}

/**
 * at_open - Called when a network interface is made active
 * @netdev: network interface device structure
 *
 * Returns 0 on success, negative value on failure
 *
 * The open entry point is called when a network interface is made
 * active by the system (IFF_UP).  At this point all resources needed
 * for transmit and receive operations are allocated, the interrupt
 * handler is registered with the OS, the watchdog timer is started,
 * and the stack is notified that the interface is ready.
 **/

static int
at_open(struct net_device *netdev)
{
    struct at_adapter *adapter = netdev_priv(netdev);
    int err;
    u32 val;

    DEBUGFUNC("at_open !");

    /* disallow open during test */
    if (test_bit(__AT_TESTING, &adapter->flags))
        return -EBUSY;

    /* allocate transmit descriptors */

    if((err = at_setup_ring_resources(adapter)))
        return err;

    if((err = at_init_hw(&adapter->hw))) {
        err = -EIO;
        goto err_init_hw;
    }
    
    /* hardware has been reset, we need to reload some things */

    at_set_multi(netdev);
    init_ring_ptrs(adapter);

#ifdef NETIF_F_HW_VLAN_TX
    at_restore_vlan(adapter);
#endif

    err = at_alloc_rx_buffers(adapter);
    
    if (0 == err) { // no RX BUFFER allocated
        err = -ENOMEM;
        goto err_nomem;
    }    
    
    if (at_configure(adapter)) {
        err = -EIO;
        goto err_config;
    }
   
    if ((err = at_request_irq(adapter)))
        goto err_req_irq;
        
    clear_bit(__AT_DOWN, &adapter->flags);     
    
#ifdef CONFIG_AT_NAPI
    netif_poll_enable(netdev);
#endif
      
    mod_timer(&adapter->watchdog_timer, jiffies + 4*HZ); 
    
    val = AT_READ_REG(&adapter->hw, REG_MASTER_CTRL);
    AT_WRITE_REG(&adapter->hw, REG_MASTER_CTRL, val|MASTER_CTRL_MANUAL_INT);
    
    
    at_irq_enable(adapter);

    return 0;

err_init_hw:
err_req_irq:
err_config:
err_nomem:
    at_free_ring_resources(adapter);
    at_reset_hw(&adapter->hw);
    
    return err;
}

/**
 * at_close - Disables a network interface
 * @netdev: network interface device structure
 *
 * Returns 0, this is not allowed to fail
 *
 * The close entry point is called when an interface is de-activated
 * by the OS.  The hardware is still under the drivers control, but
 * needs to be disabled.  A global MAC reset is issued to stop the
 * hardware, and all transmit and receive resources are freed.
 **/

static int
at_close(struct net_device *netdev)
{
    struct at_adapter *adapter = netdev_priv(netdev);
    DEBUGFUNC("at_close!");

    WARN_ON(test_bit(__AT_RESETTING, &adapter->flags));
    
    at_down(adapter);
    at_free_irq(adapter);
    at_free_ring_resources(adapter);

    return 0;
}


/**
 * at_setup_mem_resources - allocate Tx / RX descriptor resources 
 * @adapter: board private structure
 *
 * Return 0 on success, negative on failure
 **/

s32
at_setup_ring_resources(struct at_adapter *adapter)
{
    struct at_tpd_ring *tpd_ring = &adapter->tpd_ring;
    struct at_rfd_ring *rfd_ring = &adapter->rfd_ring;
    struct at_rrd_ring *rrd_ring = &adapter->rrd_ring;
    struct at_ring_header * ring_header = &adapter->ring_header;
    struct pci_dev *pdev = adapter->pdev;
    int size;
    u8 offset = 0;

//    DEBUGFUNC("at_setup_ring_resources !");

//    DEBUGOUT1("TPD count = %x  RFD cont = %x  RRD = %x", 
//        tpd_ring->count, rfd_ring->count, rrd_ring->count);
        
    DEBUGOUT1("sz TPD:%d, sz RFD:%d, sz RRD:%d", 
            sizeof(TpdDescr),
            sizeof(rx_free_desc_t),
            sizeof(rx_return_desc_t));

    size = sizeof(struct at_buffer) * (tpd_ring->count+rfd_ring->count);
    tpd_ring->buffer_info = kmalloc(size, GFP_KERNEL);
    if(!tpd_ring->buffer_info) {
    DEBUGOUT1("kmalloc failed , size = D%d", size);
        return -ENOMEM;
    }

    rfd_ring->buffer_info = 
         (struct at_buffer*)(tpd_ring->buffer_info+tpd_ring->count);

    memset(tpd_ring->buffer_info, 0, size);

    /* real ring DMA buffer */
    ring_header->size = size =  sizeof(TpdDescr ) * tpd_ring->count 
                              + sizeof(rx_free_desc_t) *    rfd_ring->count 
                              + sizeof(rx_return_desc_t) *  rrd_ring->count
                              + sizeof(coals_msg_block_t)
                              + sizeof(stats_msg_block_t)
                              + 40; // 40: for 8 bytes align
         
    ring_header->desc = 
          pci_alloc_consistent(pdev, ring_header->size, &ring_header->dma);
    if(!ring_header->desc) {
        kfree(tpd_ring->buffer_info);
        DEBUGOUT1("pci_alloc_consistent failed , size = D%d", size);
        return -ENOMEM;
    }
    

    if (adapter->pci_using_64) 
    { // test whether HIDWORD dma buffer is not cross boundary
        if (    ((ring_header->dma       &0xffffffff00000000ULL)>>32)
             != (((ring_header->dma+size)&0xffffffff00000000ULL)>>32) )
        {
            kfree(tpd_ring->buffer_info);
            pci_free_consistent(
                     pdev, 
                     ring_header->size, 
                     ring_header->desc, 
                     ring_header->dma);
            DEBUGOUT("memory allocated cross 32bit boundary !");
            return -ENOMEM;
        }
    }

//    DEBUGOUT("memory allocated successfully !");    
    
    memset(ring_header->desc, 0, ring_header->size);
    DEBUGOUT1("whole ring dma addr=%x", ring_header->dma);

    // init TPD ring
    tpd_ring->dma = ring_header->dma;
    offset = (tpd_ring->dma & 0x7) ? (8 - (ring_header->dma & 0x7)) : 0;
    tpd_ring->dma += offset;
    tpd_ring->desc = (u8*) ring_header->desc + offset;
    tpd_ring->size = sizeof(TpdDescr) * tpd_ring->count;
    DEBUGOUT1("tpd ring dma addr=%x", tpd_ring->dma);
    
    // init RFD ring
    rfd_ring->dma = tpd_ring->dma + tpd_ring->size;
    offset = (rfd_ring->dma & 0x7) ? (8 - (rfd_ring->dma & 0x7)) : 0;
    rfd_ring->dma += offset;
    rfd_ring->desc = (u8*) tpd_ring->desc + (tpd_ring->size+offset);
    rfd_ring->size = sizeof(rx_free_desc_t) * rfd_ring->count;
    DEBUGOUT1("rfd ring dma addr=%x", rfd_ring->dma);
    
    // init RRD ring
    rrd_ring->dma = rfd_ring->dma + rfd_ring->size;
    offset = (rrd_ring->dma & 0x7) ? (8 - (rrd_ring->dma & 0x7)) : 0;
    rrd_ring->dma += offset;
    rrd_ring->desc = (u8*) rfd_ring->desc + (rfd_ring->size+offset);
    rrd_ring->size = sizeof(rx_return_desc_t) * rrd_ring->count;
    DEBUGOUT1("rrd ring dma addr=%x", rrd_ring->dma);
    
    // init CMB
    adapter->cmb.dma = rrd_ring->dma + rrd_ring->size;
    offset = (adapter->cmb.dma & 0x7)? (8-(adapter->cmb.dma & 0x7)) : 0;
    adapter->cmb.dma += offset;
    adapter->cmb.cmb = 
       (coals_msg_block_t*)
          ((u8*)rrd_ring->desc + (rrd_ring->size+offset));
    DEBUGOUT1("cmd dma addr=%x", adapter->cmb.dma);
    
    // init SMB
    adapter->smb.dma = adapter->cmb.dma + sizeof(coals_msg_block_t);
    offset = (adapter->smb.dma&0x7) ? (8-(adapter->smb.dma&0x7)): 0;
    adapter->smb.dma += offset;
    adapter->smb.smb = 
        (stats_msg_block_t*)
          ((u8*)adapter->cmb.cmb + (sizeof(coals_msg_block_t)+offset));
    DEBUGOUT1("smb dma addr=%x", adapter->smb.dma);

    return AT_SUCCESS;
}


void
init_ring_ptrs(struct at_adapter *adapter)
{
    struct at_tpd_ring *tpd_ring = &adapter->tpd_ring;
    struct at_rfd_ring *rfd_ring = &adapter->rfd_ring;
    struct at_rrd_ring *rrd_ring = &adapter->rrd_ring;
    
    // Read / Write Ptr Initialize:
    atomic_set(&tpd_ring->next_to_use, 0);
    atomic_set(&tpd_ring->next_to_clean, 0);
    
    rfd_ring->next_to_clean = 0;
    atomic_set(&rfd_ring->next_to_use, 0);
    
    rrd_ring->next_to_use = 0;
    atomic_set(&rrd_ring->next_to_clean, 0);
}


/**
 * at_free_ring_resources - Free Tx / RX descriptor Resources
 * @adapter: board private structure
 *
 * Free all transmit software resources
 **/

void
at_free_ring_resources(struct at_adapter *adapter)
{
    struct pci_dev *pdev = adapter->pdev;
    struct at_tpd_ring *tpd_ring = &adapter->tpd_ring; 
    struct at_rfd_ring *rfd_ring = &adapter->rfd_ring;
    struct at_rrd_ring * rrd_ring = &adapter->rrd_ring;
    struct at_ring_header * ring_header = &adapter->ring_header;
    
//    DEBUGFUNC("at_free_ring_resources !");

    at_clean_tx_ring(adapter);
    at_clean_rx_ring(adapter);
    
    if (tpd_ring->buffer_info) {
        kfree(tpd_ring->buffer_info);
    }
    
    if (ring_header->desc) {
        pci_free_consistent(
            pdev, 
            ring_header->size, 
            ring_header->desc, 
            ring_header->dma);
    }
    
    tpd_ring->buffer_info = NULL;
    tpd_ring->desc = NULL;
    tpd_ring->dma = 0;
    
    rfd_ring->buffer_info = NULL;
    rfd_ring->desc = NULL;
    rfd_ring->dma = 0;
    
    rrd_ring->desc = NULL;
    rrd_ring->dma = 0;
}

int
at_up(struct at_adapter *adapter)
{
    struct net_device *netdev = adapter->netdev;
    int err = 0;
    u32 val;

    DEBUGFUNC("at_up !"); 

    /* hardware has been reset, we need to reload some things */

    err = at_init_hw(&adapter->hw);
    if (err) {
        err = -EIO;
        return err;
    }

    at_set_multi(netdev);
    init_ring_ptrs(adapter);

#ifdef NETIF_F_HW_VLAN_TX
    at_restore_vlan(adapter);
#endif

    err = at_alloc_rx_buffers(adapter);
    if (0 == err) { // no RX BUFFER allocated
        return -ENOMEM;
    }    
    
    if (at_configure(adapter)) {
        err = -EIO;
        goto err_up;
    }    

    clear_bit(__AT_DOWN, &adapter->flags);
    
    val = AT_READ_REG(&adapter->hw, REG_MASTER_CTRL);
    AT_WRITE_REG(&adapter->hw, REG_MASTER_CTRL, val|MASTER_CTRL_MANUAL_INT);

#ifdef CONFIG_AT_NAPI
    netif_poll_enable(netdev);
#endif

    at_irq_enable(adapter);
err_up:
    return err;
}

inline void
at_setup_mac_ctrl(struct at_adapter* adapter)
{
    u32 value;
    struct at_hw* hw = &adapter->hw;
    struct net_device* netdev = adapter->netdev;
    
    /* Config MAC CTRL Register */
    value = MAC_CTRL_TX_EN | 
            MAC_CTRL_RX_EN ;
    // duplex
    if (FULL_DUPLEX == adapter->link_duplex)
        value |= MAC_CTRL_DUPLX;
    // speed
    value |= ((u32)((SPEED_1000 == adapter->link_speed) ?
                 MAC_CTRL_SPEED_1000:MAC_CTRL_SPEED_10_100)<<MAC_CTRL_SPEED_SHIFT);
    // flow control
    value |= (MAC_CTRL_TX_FLOW|MAC_CTRL_RX_FLOW);

    // PAD & CRC
    value |= (MAC_CTRL_ADD_CRC|MAC_CTRL_PAD);
    // preamble length
    value |= (((u32)adapter->hw.preamble_len
                  &MAC_CTRL_PRMLEN_MASK)<< MAC_CTRL_PRMLEN_SHIFT);
    // vlan 
    if (adapter->vlgrp)     
        value |= MAC_CTRL_RMV_VLAN;
        
    // filter mode
    value |= MAC_CTRL_BC_EN;
    if (netdev->flags & IFF_PROMISC) 
        value |= MAC_CTRL_PROMIS_EN;
    else if (netdev->flags & IFF_ALLMULTI)
        value |= MAC_CTRL_MC_ALL_EN;

    AT_WRITE_REG(hw, REG_MAC_CTRL, value);
}


static int
at_check_link(struct at_adapter* adapter)
{
    struct at_hw *hw = &adapter->hw;
    struct net_device * netdev = adapter->netdev;
    int ret_val;
    u16 speed, duplex, phy_data;
    int reconfig = 0;

    DEBUGFUNC("at_check_link !");

    // MII_BMSR must read twise
    at_read_phy_reg(hw, MII_BMSR, &phy_data);
    at_read_phy_reg(hw, MII_BMSR, &phy_data);
    if (!(phy_data&BMSR_LSTATUS)) { // link down
        if (netif_carrier_ok(netdev)) { // old link state: Up
            u32 value;
            DEBUGOUT("NIC Link is Down");
            //disable rx
            value = AT_READ_REG(hw, REG_MAC_CTRL);
            value &= ~MAC_CTRL_RX_EN;
            AT_WRITE_REG(hw, REG_MAC_CTRL, value);            
            adapter->link_speed = SPEED_0;
            clear_bit(0, &hw->force_ps);
            netif_carrier_off(netdev);
            netif_stop_queue(netdev);
        }
        return AT_SUCCESS;  
    }
    
    // Link Up
    ret_val = at_get_speed_and_duplex(hw, &speed, &duplex);
    clear_bit(0, &hw->force_ps);
    if (ret_val)  return ret_val;
    switch( hw->MediaType )
    {
    case MEDIA_TYPE_100M_FULL:
        if (speed  != SPEED_100 || duplex != FULL_DUPLEX)
            reconfig = 1;
        break;
    case MEDIA_TYPE_100M_HALF:
        if (speed  != SPEED_100 || duplex != HALF_DUPLEX)
            reconfig = 1;
        break;
    case MEDIA_TYPE_10M_FULL:
        if (speed != SPEED_10 || duplex != FULL_DUPLEX)
            reconfig = 1;
            break;  
    case MEDIA_TYPE_10M_HALF:
        if (speed  != SPEED_10 || duplex != HALF_DUPLEX)
            reconfig = 1;
        break;
    }
    // link result is our setting
    if (0 == reconfig)
    {
        if (adapter->link_speed != speed ||
            adapter->link_duplex != duplex ) {
            adapter->link_speed = speed;
            adapter->link_duplex = duplex;
            at_setup_mac_ctrl(adapter); 
            printk(KERN_INFO
                   "%s: %s NIC Link is Up<%d Mbps %s>\n",
                    at_driver_name,
                    netdev->name, adapter->link_speed,
                    adapter->link_duplex == FULL_DUPLEX ?
                    "Full Duplex" : "Half Duplex"); 
        }
        
        if (!netif_carrier_ok(netdev)) { // Link down -> Up
            netif_carrier_on(netdev);
            netif_wake_queue(netdev);
        }
        return AT_SUCCESS;
    }
    
    // change orignal link status
    if (netif_carrier_ok(netdev)) {
        u32 value;
        // disable rx
        value = AT_READ_REG(hw, REG_MAC_CTRL);
        value &= ~MAC_CTRL_RX_EN;
        AT_WRITE_REG(hw, REG_MAC_CTRL, value); 
            
        adapter->link_speed = SPEED_0;
        netif_carrier_off(netdev);
        netif_stop_queue(netdev);
    }
    
    // auto-neg, insert timer to re-config phy (if interval smaller than 5 seconds, something strange)
    if (!test_and_set_bit(0, &adapter->cfg_phy)) {
        mod_timer(&adapter->phy_config_timer, jiffies + 5 * HZ);
    }

    return AT_SUCCESS;
}

void
at_down(struct at_adapter *adapter)
{
    struct net_device *netdev = adapter->netdev;
    
    DEBUGFUNC("at_down !");

     /* signal that we're down so the interrupt handler does not
      * reschedule our watchdog timer */
    set_bit(__AT_DOWN, &adapter->flags);

#ifdef NETIF_F_LLTX
    netif_stop_queue(netdev);
#else
    netif_tx_disable(netdev);
#endif

    /* reset MAC to disable all RX/TX */
    at_reset_hw(&adapter->hw);
    adapter->cmb.cmb->int_stats = 0;
    msleep(1);
    
#ifdef CONFIG_AT_NAPI
    netif_poll_disable(netdev);
#endif    
    at_irq_disable(adapter);
    
    del_timer_sync(&adapter->watchdog_timer);
    del_timer_sync(&adapter->phy_config_timer);
    clear_bit(0, &adapter->cfg_phy);

    
    netif_carrier_off(netdev);
    adapter->link_speed = SPEED_0;
    adapter->link_duplex = -1;
    
    at_clean_tx_ring(adapter);
    at_clean_rx_ring(adapter);
}

/**
 * at_set_multi - Multicast and Promiscuous mode set
 * @netdev: network interface device structure
 *
 * The set_multi entry point is called whenever the multicast address
 * list or the network interface flags are updated.  This routine is
 * responsible for configuring the hardware for proper multicast,
 * promiscuous mode, and all-multi behavior.
 **/

static void
at_set_multi(struct net_device *netdev)
{
    struct at_adapter *adapter = netdev_priv(netdev);
    struct at_hw *hw = &adapter->hw;
    struct dev_mc_list *mc_ptr;
    u32 rctl;
    u32 hash_value;

    DEBUGFUNC("at_set_multi !");

    /* Check for Promiscuous and All Multicast modes */

    rctl = AT_READ_REG(hw, REG_MAC_CTRL);

    if(netdev->flags & IFF_PROMISC) {
        rctl |= MAC_CTRL_PROMIS_EN;
    } else if(netdev->flags & IFF_ALLMULTI) {
        rctl |= MAC_CTRL_MC_ALL_EN;
        rctl &= ~MAC_CTRL_PROMIS_EN;
    } else {
        rctl &= ~(MAC_CTRL_PROMIS_EN | MAC_CTRL_MC_ALL_EN);
    }

    AT_WRITE_REG(hw, REG_MAC_CTRL, rctl);

    /* clear the old settings from the multicast hash table */
    AT_WRITE_REG(hw, REG_RX_HASH_TABLE, 0);
    AT_WRITE_REG_ARRAY(hw, REG_RX_HASH_TABLE, 1, 0);

    /* comoute mc addresses' hash value ,and put it into hash table */

    for(mc_ptr = netdev->mc_list; mc_ptr; mc_ptr = mc_ptr->next) {
        hash_value = at_hash_mc_addr(hw, mc_ptr->dmi_addr);
        at_hash_set(hw, hash_value);
    }
}



#ifdef NETIF_F_HW_VLAN_TX
static void
at_vlan_rx_register(struct net_device *netdev, struct vlan_group *grp)
{
    struct at_adapter *adapter = netdev_priv(netdev);
    u32 ctrl;

 //   DEBUGFUNC("at_vlan_rx_register !");    

    at_irq_disable(adapter);
    adapter->vlgrp = grp;

    if(grp) {
        /* enable VLAN tag insert/strip */

        ctrl = AT_READ_REG(&adapter->hw, REG_MAC_CTRL);
        ctrl |= MAC_CTRL_RMV_VLAN; 
        AT_WRITE_REG(&adapter->hw, REG_MAC_CTRL, ctrl);
    } else {
        /* disable VLAN tag insert/strip */

        ctrl = AT_READ_REG(&adapter->hw, REG_MAC_CTRL);
        ctrl &= ~MAC_CTRL_RMV_VLAN;
        AT_WRITE_REG(&adapter->hw, REG_MAC_CTRL, ctrl);
    }

    at_irq_enable(adapter);
}

static void
at_restore_vlan(struct at_adapter *adapter)
{
//    DEBUGFUNC("at_restore_vlan !");
    at_vlan_rx_register(adapter->netdev, adapter->vlgrp);
}

#endif

static inline
void set_flow_ctrl_old(struct at_adapter* adapter)
{
    u32 Hi, Lo, value;
    
    // RFD Flow Control   
    value = adapter->rfd_ring.count;
    Hi = value / 16;
    if (Hi < 2) Hi = 2;
    Lo = value * 7 / 8;
    
   value = ((Hi&RXQ_RXF_PAUSE_TH_HI_MASK) << RXQ_RXF_PAUSE_TH_HI_SHIFT)|
    ((Lo&RXQ_RXF_PAUSE_TH_LO_MASK) << RXQ_RXF_PAUSE_TH_LO_SHIFT);
    AT_WRITE_REG(&adapter->hw, REG_RXQ_RXF_PAUSE_THRESH, value);
    
    // RRD Flow Control
    value = adapter->rrd_ring.count;
    Lo = value / 16;
    Hi = value * 7 / 8;
    if (Lo < 2) Lo = 2;

   value = ((Hi&RXQ_RRD_PAUSE_TH_HI_MASK) << RXQ_RRD_PAUSE_TH_HI_SHIFT)|
    ((Lo&RXQ_RRD_PAUSE_TH_LO_MASK) << RXQ_RRD_PAUSE_TH_LO_SHIFT);
    AT_WRITE_REG(&adapter->hw, REG_RXQ_RRD_PAUSE_THRESH, value);
}

static inline
void set_flow_ctrl_new(struct at_hw* hw)
{
    u32 Hi, Lo, value;
    
    // RXF Flow Control   
    value = AT_READ_REG(hw, REG_SRAM_RXF_LEN);
    Lo = value / 16;
    if (Lo < 192) Lo = 192;
    Hi = value * 7 / 8;
    if (Hi < Lo) Hi = Lo + 16;
   value = ((Hi&RXQ_RXF_PAUSE_TH_HI_MASK) << RXQ_RXF_PAUSE_TH_HI_SHIFT)|
    ((Lo&RXQ_RXF_PAUSE_TH_LO_MASK) << RXQ_RXF_PAUSE_TH_LO_SHIFT);
    AT_WRITE_REG(hw, REG_RXQ_RXF_PAUSE_THRESH, value);
    
    // RRD Flow Control
    value = AT_READ_REG(hw, REG_SRAM_RRD_LEN);
    Lo = value / 8;
    Hi = value * 7 / 8;
    if (Lo < 2) Lo = 2;
    if (Hi < Lo) Hi = Lo + 3;
   value = ((Hi&RXQ_RRD_PAUSE_TH_HI_MASK) << RXQ_RRD_PAUSE_TH_HI_SHIFT)|
    ((Lo&RXQ_RRD_PAUSE_TH_LO_MASK) << RXQ_RRD_PAUSE_TH_LO_SHIFT);
    AT_WRITE_REG(hw, REG_RXQ_RRD_PAUSE_THRESH, value);
}

/**
 * at_configure - Configure Transmit&Receive Unit after Reset
 * @adapter: board private structure
 *
 * Configure the Tx /Rx unit of the MAC after a reset.
 **/

static s32
at_configure(struct at_adapter *adapter)
{
    struct at_hw * hw = &adapter->hw;
    u32 value;
    
//  DEBUGFUNC("at_configure !");
    // clear interrupt status
    AT_WRITE_REG(&adapter->hw, REG_ISR, 0xffffffff);

    // set MAC Address
    value = (((u32)hw->mac_addr[2]) << 24) |
            (((u32)hw->mac_addr[3]) << 16) |
            (((u32)hw->mac_addr[4]) << 8 ) |
            (((u32)hw->mac_addr[5])      ) ;
    AT_WRITE_REG(hw, REG_MAC_STA_ADDR, value);
    value = (((u32)hw->mac_addr[0]) << 8 ) |
            (((u32)hw->mac_addr[1])      ) ;
    AT_WRITE_REG(hw, (REG_MAC_STA_ADDR+4), value);

    // tx / rx ring :
    
    // HI base address
    AT_WRITE_REG(
          hw, 
          REG_DESC_BASE_ADDR_HI, 
          (u32)((adapter->tpd_ring.dma&0xffffffff00000000ULL) >>32));
    // LO base address
    AT_WRITE_REG(
          hw, 
          REG_DESC_RFD_ADDR_LO, 
          (u32)(adapter->rfd_ring.dma& 0x00000000ffffffffULL));
    AT_WRITE_REG(
          hw, 
          REG_DESC_RRD_ADDR_LO, 
          (u32)(adapter->rrd_ring.dma& 0x00000000ffffffffULL));
    AT_WRITE_REG(hw, 
                 REG_DESC_TPD_ADDR_LO, 
                 (u32)(adapter->tpd_ring.dma& 0x00000000ffffffffULL));
    AT_WRITE_REG(hw, 
                 REG_DESC_CMB_ADDR_LO, 
                 (u32)(adapter->cmb.dma& 0x00000000ffffffffULL));
    AT_WRITE_REG(hw, 
                 REG_DESC_SMB_ADDR_LO, 
                 (u32)(adapter->smb.dma& 0x00000000ffffffffULL));
    
    // element count
    value = adapter->rrd_ring.count;
    value <<= 16;
    value += adapter->rfd_ring.count;
    AT_WRITE_REG(hw, REG_DESC_RFD_RRD_RING_SIZE, value);
    AT_WRITE_REG(hw, REG_DESC_TPD_RING_SIZE, adapter->tpd_ring.count);

/* 
    // config SRAM
    // add RXF 256*8 bytes   
    value = ((2795 + 256) << 16) | 432;
    AT_WRITE_REG(hw, REG_SRAM_RXF_ADDR, value);
    value = 2364 + 256;
    AT_WRITE_REG(hw, REG_SRAM_RXF_LEN, value);
    // sub TXF 256*8 bytes
    value = (4075 << 16) | (2796 + 256);
    AT_WRITE_REG(hw, REG_SRAM_TXF_ADDR, value);
    value = 1280 - 256;
    AT_WRITE_REG(hw, REG_SRAM_TXF_LEN, value);
*/
    // Load Ptr
    AT_WRITE_REG(hw, REG_LOAD_PTR, 1);
    
    
    /* config Mailbox */
    
    value = 
         (((u32)atomic_read(&adapter->tpd_ring.next_to_use)
            &MB_TPD_PROD_INDX_MASK)<<MB_TPD_PROD_INDX_SHIFT) |
         (((u32)atomic_read(&adapter->rrd_ring.next_to_clean)
            &MB_RRD_CONS_INDX_MASK)<<MB_RRD_CONS_INDX_SHIFT) |
         (((u32)atomic_read(&adapter->rfd_ring.next_to_use)
            &MB_RFD_PROD_INDX_MASK)<<MB_RFD_PROD_INDX_SHIFT);
    AT_WRITE_REG(hw, REG_MAILBOX, value);
    
//    DEBUGOUT1("init Mailbox with 0x%x", value);
    
    /* config IPG/IFG */
    value = 
        (((u32)hw->ipgt&MAC_IPG_IFG_IPGT_MASK) 
              <<MAC_IPG_IFG_IPGT_SHIFT) |
        (((u32)hw->min_ifg &MAC_IPG_IFG_MIFG_MASK) 
              <<MAC_IPG_IFG_MIFG_SHIFT) |
        (((u32)hw->ipgr1&MAC_IPG_IFG_IPGR1_MASK)
              <<MAC_IPG_IFG_IPGR1_SHIFT)|
        (((u32)hw->ipgr2&MAC_IPG_IFG_IPGR2_MASK)
              <<MAC_IPG_IFG_IPGR2_SHIFT);
    AT_WRITE_REG(hw, REG_MAC_IPG_IFG, value);
//    DEBUGOUT1("init ipg/ifg with 0x%x", value);
    
    /* config  Half-Duplex Control */
    value = 
      ((u32)hw->lcol&MAC_HALF_DUPLX_CTRL_LCOL_MASK) |
      (((u32)hw->max_retry&MAC_HALF_DUPLX_CTRL_RETRY_MASK)
          <<MAC_HALF_DUPLX_CTRL_RETRY_SHIFT) |
      MAC_HALF_DUPLX_CTRL_EXC_DEF_EN   |
      (0xa<<MAC_HALF_DUPLX_CTRL_ABEBT_SHIFT) |
      (((u32)hw->jam_ipg&MAC_HALF_DUPLX_CTRL_JAMIPG_MASK)
          <<MAC_HALF_DUPLX_CTRL_JAMIPG_SHIFT);
    AT_WRITE_REG(hw, REG_MAC_HALF_DUPLX_CTRL, value);
//    DEBUGOUT1("init Half Duplex with 0x%x", value);
    
    
    /* set Interrupt Moderator Timer */
    AT_WRITE_REGW(hw, REG_IRQ_MODU_TIMER_INIT, adapter->imt);
    AT_WRITE_REG(hw, REG_MASTER_CTRL, MASTER_CTRL_ITIMER_EN);
//    DEBUGOUT1("init Irq Modurator Timer with 0x%x", adapter->imt);
    
    /* set Interrupt Clear Timer */
    AT_WRITE_REGW(hw, REG_CMBDISDMA_TIMER, adapter->ict);
//    DEBUGOUT1("init Irq Clear Timer with 0x%x", adapter->ict);
    
    /* set MTU */
    AT_WRITE_REG(hw, REG_MTU, hw->max_frame_size+
                                ENET_HEADER_SIZE + 
                                VLAN_SIZE +
                                ETHERNET_FCS_SIZE );
    DEBUGOUT1("init MTU with 0x%x", hw->max_frame_size); 

    // jumbo size & rrd retirement timer
    value = 
      (((u32)hw->rx_jumbo_th&RXQ_JMBOSZ_TH_MASK) 
          << RXQ_JMBOSZ_TH_SHIFT)|
      (((u32)hw->rx_jumbo_lkah&RXQ_JMBO_LKAH_MASK) 
          << RXQ_JMBO_LKAH_SHIFT)|
      (((u32)hw->rrd_ret_timer&RXQ_RRD_TIMER_MASK)
          << RXQ_RRD_TIMER_SHIFT) ;
    AT_WRITE_REG(hw, REG_RXQ_JMBOSZ_RRDTIM, value);
//    DEBUGOUT1("init RXQ Jumbo size RRD retirement Timer with 0x=%x", value);
    // Flow Control
    switch (hw->dev_rev)        
    {
    case 0x8001: 
    case 0x9001:
    case 0x9002:
    case 0x9003:
        set_flow_ctrl_old(adapter);
        break;
    default:
        set_flow_ctrl_new(hw);
        break;
    }
                
    /* config TXQ */
    value =  
       (((u32)hw->tpd_burst&TXQ_CTRL_TPD_BURST_NUM_MASK)
           << TXQ_CTRL_TPD_BURST_NUM_SHIFT) |
       (((u32)hw->txf_burst&TXQ_CTRL_TXF_BURST_NUM_MASK) 
           << TXQ_CTRL_TXF_BURST_NUM_SHIFT) |
       (((u32)hw->tpd_fetch_th&TXQ_CTRL_TPD_FETCH_TH_MASK)
           << TXQ_CTRL_TPD_FETCH_TH_SHIFT) |
       TXQ_CTRL_ENH_MODE |
       TXQ_CTRL_EN;
    AT_WRITE_REG(hw, REG_TXQ_CTRL, value);
//    DEBUGOUT1("init TXQ Control with 0x%x", value); 
     
    // min tpd fetch gap & tx jumbo packet size threshold for taskoffload
    value = 
      (((u32)hw->tx_jumbo_task_th&TX_JUMBO_TASK_TH_MASK)
         << TX_JUMBO_TASK_TH_SHIFT) |
      (((u32)hw->tpd_fetch_gap&TX_TPD_MIN_IPG_MASK)
         << TX_TPD_MIN_IPG_SHIFT);
    AT_WRITE_REG(hw, REG_TX_JUMBO_TASK_TH_TPD_IPG, value);
//    DEBUGOUT1("init TPD fetch gap & TX jumbo taskoffload threshold with 0x%x", value);
    
    /* config RXQ */
    value =  
      (((u32)hw->rfd_burst&RXQ_CTRL_RFD_BURST_NUM_MASK) 
          << RXQ_CTRL_RFD_BURST_NUM_SHIFT)|
      (((u32)hw->rrd_burst&RXQ_CTRL_RRD_BURST_THRESH_MASK)
          << RXQ_CTRL_RRD_BURST_THRESH_SHIFT)|
      (((u32)hw->rfd_fetch_gap&RXQ_CTRL_RFD_PREF_MIN_IPG_MASK)
          << RXQ_CTRL_RFD_PREF_MIN_IPG_SHIFT) |
      RXQ_CTRL_CUT_THRU_EN | 
      RXQ_CTRL_EN ;
      AT_WRITE_REG(hw, REG_RXQ_CTRL, value);
//     DEBUGOUT1("init RXQ control with 0x%x", value);
   
    /* config  DMA Engine */
    {
        u32 v1;
        u16 max_dmar, max_dmaw, l;
        max_dmar = hw->dmar_block;
        max_dmaw = hw->dmaw_block;
        v1 = AT_READ_REGW(hw, REG_DEVICE_CTRL);
        l = (v1>>DEVICE_CTRL_MAX_PAYLOAD_SHIFT)&DEVICE_CTRL_MAX_PAYLOAD_MASK;
        if (l < max_dmaw)
            max_dmaw = l;
        l = (v1>>DEVICE_CTRL_MAX_RREQ_SZ_SHIFT)&DEVICE_CTRL_MAX_RREQ_SZ_MASK;
        if (l < max_dmar)
            max_dmar = l;
        hw->dmar_block = max_dmar;
        hw->dmaw_block = max_dmaw;
    }      
    
    value = 
      ((((u32)hw->dmar_block)&DMA_CTRL_DMAR_BURST_LEN_MASK) 
           << DMA_CTRL_DMAR_BURST_LEN_SHIFT)|
      ((((u32)hw->dmaw_block)&DMA_CTRL_DMAW_BURST_LEN_MASK) 
           << DMA_CTRL_DMAW_BURST_LEN_SHIFT) |
      DMA_CTRL_DMAR_EN | 
      DMA_CTRL_DMAW_EN;
    value  |= (u32)hw->dma_ord;
    if (at_rcb_128 == hw->rcb_value) 
    {
        value |= DMA_CTRL_RCB_VALUE;
    }
    AT_WRITE_REG(hw, REG_DMA_CTRL, value);
//    DEBUGOUT1("init DMA Engine with 0x%x", value);
    
    /* config CMB / SMB */
    value = (hw->cmb_tpd > adapter->tpd_ring.count) ?
             hw->cmb_tpd : adapter->tpd_ring.count;
    value <<= 16;
    value |= hw->cmb_rrd;
    AT_WRITE_REG(hw, REG_CMB_WRITE_TH, value);
    value = hw->cmb_rx_timer | ((u32)hw->cmb_tx_timer << 16);
    AT_WRITE_REG(hw, REG_CMB_WRITE_TIMER, value);
    AT_WRITE_REG(hw, REG_SMB_TIMER, hw->smb_timer);
//    DEBUGOUT1("init CMB Write Timer with 0x%x", value);

    // --- enable CMB / SMB 
    value = CSMB_CTRL_CMB_EN | CSMB_CTRL_SMB_EN;
    AT_WRITE_REG(hw, REG_CSMB_CTRL, value);
    
    value = AT_READ_REG(&adapter->hw, REG_ISR);
    if ((value&ISR_PHY_LINKDOWN) != 0) {
        value = 1; // config failed 
    } else {
        value = 0;
    }
    // clear all interrupt status
    AT_WRITE_REG(&adapter->hw, REG_ISR, 0x3fffffff);
    AT_WRITE_REG(&adapter->hw, REG_ISR, 0);
    return value;
}

/**
 * at_set_mac - Change the Ethernet Address of the NIC
 * @netdev: network interface device structure
 * @p: pointer to an address structure
 *
 * Returns 0 on success, negative on failure
 **/

static int
at_set_mac(struct net_device *netdev, void *p)
{
    struct at_adapter *adapter = netdev_priv(netdev);
    struct sockaddr *addr = p;

    DEBUGFUNC("at_set_mac !");
    
    if (!is_valid_ether_addr(addr->sa_data))
        return -EADDRNOTAVAIL;
        
    if (netif_running(netdev))
        return -EBUSY; 

    memcpy(netdev->dev_addr, addr->sa_data, netdev->addr_len);
    memcpy(adapter->hw.mac_addr, addr->sa_data, netdev->addr_len);
    
    set_mac_addr(&adapter->hw);


    return 0;
}


/**
 * at_change_mtu - Change the Maximum Transfer Unit
 * @netdev: network interface device structure
 * @new_mtu: new value for maximum frame size
 *
 * Returns 0 on success, negative on failure
 **/

static int
at_change_mtu(struct net_device *netdev, int new_mtu)
{
    struct at_adapter *adapter = netdev_priv(netdev);
    struct at_hw* hw = &adapter->hw;
    
    DEBUGFUNC("at_change_mtu !");


    if ((new_mtu < 40) || (new_mtu > (ETH_DATA_LEN + VLAN_SIZE)))
        return -EINVAL;
        

    /* set MTU */
    if (hw->max_frame_size != new_mtu) {
        
        while (test_and_set_bit(__AT_RESETTING, &adapter->flags))
            msleep(1);
        
        if (netif_running(netdev)) {
            at_down(adapter);
        }
            
        netdev->mtu = new_mtu;   
        hw->max_frame_size = new_mtu;
        hw->tx_jumbo_task_th = (new_mtu+
                                ENET_HEADER_SIZE + 
                                VLAN_SIZE +
                                ETHERNET_FCS_SIZE + 7)>>3;
        adapter->rx_buffer_len = (new_mtu + 
                                ENET_HEADER_SIZE + 
                                VLAN_SIZE +
                                ETHERNET_FCS_SIZE + 7)&~7; 
        hw->rx_jumbo_th = adapter->rx_buffer_len  / 8;


        if (netif_running(netdev))
            at_up(adapter);
        else
            at_reset(adapter);

        clear_bit(__AT_RESETTING, &adapter->flags);
    }

    return 0;
}


/**
 * at_alloc_rx_buffers - Replace used receive buffers
 * @adapter: address of board private structure
 **/

static u16
at_alloc_rx_buffers(struct at_adapter *adapter)
{
    struct at_rfd_ring *rfd_ring = &adapter->rfd_ring;
    struct net_device *netdev = adapter->netdev;
    struct pci_dev *pdev = adapter->pdev;
    struct page *page;                                                       
    unsigned long offset;                                                    
    struct at_buffer *buffer_info, * next_info;
    struct sk_buff *skb;
    u16 num_alloc = 0;
    u16 rfd_next_to_use, next_next;
    rx_free_desc_t *rfd_desc;

    next_next = rfd_next_to_use = (u16)atomic_read(&rfd_ring->next_to_use);
    if (++next_next == rfd_ring->count)      next_next = 0;
    buffer_info = &rfd_ring->buffer_info[rfd_next_to_use];
    next_info = &rfd_ring->buffer_info[next_next];

    while (!buffer_info->alloced && !next_info->alloced) {
        if (NULL != buffer_info->skb) {
            buffer_info->alloced = 1;
            DEBUGOUT1("skip rfd allocate (%d)", rfd_next_to_use);
            goto next;
        }
    
        rfd_desc = AT_RFD_DESC(rfd_ring, rfd_next_to_use);

        skb = dev_alloc_skb(adapter->rx_buffer_len + NET_IP_ALIGN);

        if(!skb) {
            /* Better luck next round */
            adapter->net_stats.rx_dropped++;
            break;
        }

        /* Make buffer alignment 2 beyond a 16 byte boundary
         * this will result in a 16 byte aligned IP header after
         * the 14 byte MAC header is removed
         */
        skb_reserve(skb, NET_IP_ALIGN);

        skb->dev = netdev;

        buffer_info->alloced = 1;
        buffer_info->skb = skb;
        buffer_info->length = (u16)adapter->rx_buffer_len;
        page = virt_to_page(skb->data);                                            
        offset = (unsigned long) skb->data & ~PAGE_MASK;  
        buffer_info->dma =
                pci_map_page(pdev,
                             page,
                             offset,
                             adapter->rx_buffer_len,
                             PCI_DMA_FROMDEVICE);
        rfd_desc->buffer_addr = cpu_to_le64(buffer_info->dma);
        rfd_desc->buf_len = cpu_to_le16(adapter->rx_buffer_len);
        rfd_desc->coalese = 0;
        
next:
        rfd_next_to_use = next_next;
        if (++next_next == rfd_ring->count)     next_next = 0;
    
        buffer_info = &rfd_ring->buffer_info[rfd_next_to_use];
        next_info = &rfd_ring->buffer_info[next_next];
        num_alloc++;
    }
    
    if (0 != num_alloc) {
        /* Force memory writes to complete before letting h/w
         * know there are new descriptors to fetch.  (Only
         * applicable for weak-ordered memory model archs,
         * such as IA-64). */
        wmb();
        atomic_set(&rfd_ring->next_to_use, (int)rfd_next_to_use);
    }
    return num_alloc;
}


void
at_read_pci_cfg(struct at_hw *hw, u32 reg, u16 *value)
{
    struct at_adapter *adapter = hw->back;

    pci_read_config_word(adapter->pdev, reg, value);
}

void
at_write_pci_cfg(struct at_hw *hw, u32 reg, u16 *value)
{
    struct at_adapter *adapter = hw->back;

    pci_write_config_word(adapter->pdev, reg, *value);
}

/**
 * at_clean_tx_ring - Free Tx Buffers
 * @adapter: board private structure
 **/

static void
at_clean_tx_ring(struct at_adapter *adapter)
{
    struct at_tpd_ring *tpd_ring = &adapter->tpd_ring;
    struct at_buffer *buffer_info;
    struct pci_dev *pdev = adapter->pdev;
    unsigned long size;
    unsigned int i;

//    DEBUGFUNC("at_clean_tx_ring !");

    if (NULL == tpd_ring->buffer_info ||
        NULL == tpd_ring->desc)
        return;
        
    /* Free all the Tx ring sk_buffs */

    for(i = 0; i < tpd_ring->count; i++) {
        buffer_info = &tpd_ring->buffer_info[i];
        if (buffer_info->dma) {
            pci_unmap_page(pdev,
                           buffer_info->dma,
                           buffer_info->length,
                           PCI_DMA_TODEVICE);
            buffer_info->dma = 0;
        }
    }
    
    for(i = 0; i < tpd_ring->count; i++) {
        buffer_info = &tpd_ring->buffer_info[i];
        if(buffer_info->skb) {
            dev_kfree_skb_any(buffer_info->skb);
            buffer_info->skb = NULL;
        }
    }

    size = sizeof(struct at_buffer) * tpd_ring->count;
    memset(tpd_ring->buffer_info, 0, size);

    /* Zero out the descriptor ring */

    memset(tpd_ring->desc, 0, tpd_ring->size);

    atomic_set(&tpd_ring->next_to_use, 0);
    atomic_set(&tpd_ring->next_to_clean, 0);
}

/**
 * at_clean_rx_ring - Free RFD Buffers
 * @adapter: board private structure
 **/

static void
at_clean_rx_ring(struct at_adapter *adapter)
{
    struct at_rfd_ring *rfd_ring = &adapter->rfd_ring;
    struct at_rrd_ring *rrd_ring = &adapter->rrd_ring;
    struct at_buffer *buffer_info;
    struct pci_dev *pdev = adapter->pdev;
    unsigned long size;
    unsigned int i;

//    DEBUGFUNC("at_clean_rx_ring !");

    if (NULL == rfd_ring->buffer_info ||
        NULL == rfd_ring->desc)
        return;
        
    /* Free all the Rx ring sk_buffs */

    for(i = 0; i < rfd_ring->count; i++) {
        buffer_info = &rfd_ring->buffer_info[i];
        if(buffer_info->dma) {

            pci_unmap_page(pdev,
                           buffer_info->dma,
                           buffer_info->length,
                           PCI_DMA_FROMDEVICE);
            buffer_info->dma = 0;
        }
        if(buffer_info->skb) {
            dev_kfree_skb(buffer_info->skb);
            buffer_info->skb = NULL;
        }
    }

    size = sizeof(struct at_buffer) * rfd_ring->count;
    memset(rfd_ring->buffer_info, 0, size);

    /* Zero out the descriptor ring */

    memset(rfd_ring->desc, 0, rfd_ring->size);

    rfd_ring->next_to_clean = 0;
    atomic_set(&rfd_ring->next_to_use, 0);
    
    rrd_ring->next_to_use = 0;
    atomic_set(&rrd_ring->next_to_clean, 0);
}

/**
 * at_get_stats - Get System Network Statistics
 * @netdev: network interface device structure
 *
 * Returns the address of the device statistics structure.
 * The statistics are actually updated from the timer callback.
 **/

static struct net_device_stats *
at_get_stats(struct net_device *netdev)
{
    struct at_adapter *adapter = netdev_priv(netdev);
  
   // DEBUGFUNC("at_get_stats");
    return &adapter->net_stats;
}       


/**
 * at_ioctl -
 * @netdev:
 * @ifreq:
 * @cmd:
 **/

static int
at_ioctl(struct net_device *netdev, struct ifreq *ifr, int cmd)
{
    DEBUGFUNC("at_ioctl !");
    switch (cmd) {
#ifdef SIOCGMIIPHY
    case SIOCGMIIPHY:
    case SIOCGMIIREG:
    case SIOCSMIIREG:
        return at_mii_ioctl(netdev, ifr, cmd);
#endif

#ifdef ETHTOOL_OPS_COMPAT
    case SIOCETHTOOL:
        return ethtool_ioctl(ifr);
#endif

    default:
        return -EOPNOTSUPP;
    }
}


#ifdef SIOCGMIIPHY
/**
 * at_mii_ioctl -
 * @netdev:
 * @ifreq:
 * @cmd:
 **/

static int
at_mii_ioctl(struct net_device *netdev, struct ifreq *ifr, int cmd)
{
    struct at_adapter *adapter = netdev_priv(netdev);
    struct mii_ioctl_data *data = if_mii(ifr);
    unsigned long flags;
    
    DEBUGFUNC("at_mii_ioctl !");

    switch (cmd) {
    case SIOCGMIIPHY:
        data->phy_id = 0;
        break;
    case SIOCGMIIREG:
        if (!capable(CAP_NET_ADMIN))
            return -EPERM;
        spin_lock_irqsave(&adapter->stats_lock, flags);
        if (at_read_phy_reg(&adapter->hw, data->reg_num & 0x1F, &data->val_out)) {
            spin_unlock_irqrestore(&adapter->stats_lock, flags);
            return -EIO;
        }
        spin_unlock_irqrestore(&adapter->stats_lock, flags);
        break;
    case SIOCSMIIREG:
        if (!capable(CAP_NET_ADMIN))
            return -EPERM;
        if (data->reg_num & ~(0x1F))
            return -EFAULT;
         
        spin_lock_irqsave(&adapter->stats_lock, flags);
        DEBUGOUT1("<at_mii_ioctl> write %x %x", 
            data->reg_num, 
            data->val_in);
        if (at_write_phy_reg(&adapter->hw, data->reg_num, data->val_in)) {
            spin_unlock_irqrestore(&adapter->stats_lock, flags);
            return -EIO;
        }
        // ......
        spin_unlock_irqrestore(&adapter->stats_lock, flags);
        break;
        
    default:
        return -EOPNOTSUPP;
    }
    return AT_SUCCESS;
}

#endif


/**
 * at_tx_timeout - Respond to a Tx Hang
 * @netdev: network interface device structure
 **/

static void
at_tx_timeout(struct net_device *netdev)
{
    struct at_adapter *adapter = netdev_priv(netdev);

    DEBUGFUNC("at_tx_timeout !");

    /* Do the reset outside of interrupt context */
    schedule_work(&adapter->reset_task);
}

static void
at_reset_task(struct work_struct *work)
{
    struct at_adapter *adapter;
    adapter = container_of(work, struct at_adapter, reset_task);

    at_reinit_locked(adapter);
}


void
at_reinit_locked(struct at_adapter *adapter)
{
 
    DEBUGFUNC("at_reinit_locked !");

    WARN_ON(in_interrupt());
    while (test_and_set_bit(__AT_RESETTING, &adapter->flags))
        msleep(1);
    at_down(adapter);
    at_up(adapter);
    clear_bit(__AT_RESETTING, &adapter->flags);
}


/**
 * at_link_chg_task - deal with link change event Out of interrupt context
 * @netdev: network interface device structure
 **/
static void
at_link_chg_task(struct work_struct *work)
{
    struct at_adapter *adapter;
    unsigned long flags;

    DEBUGFUNC("at_link_chg_task !");

    adapter = container_of(work, struct at_adapter, link_chg_task);


    spin_lock_irqsave(&adapter->stats_lock, flags);
    at_check_link(adapter);
    spin_unlock_irqrestore(&adapter->stats_lock, flags);
}

static void
at_check_for_link(struct at_adapter* adapter)
{
    struct net_device *netdev = adapter->netdev;
    u16 phy_data = 0;

    DEBUGFUNC("at_check_for_link!");
    
    spin_lock(&adapter->stats_lock);
    at_read_phy_reg(&adapter->hw, MII_BMSR, &phy_data);
    at_read_phy_reg(&adapter->hw, MII_BMSR, &phy_data);
    spin_unlock(&adapter->stats_lock);
    
    DEBUGOUT1("MII_BMSR=%x <at_check_for_link>", phy_data);
    
    // notify upper layer link down ASAP
    if (!(phy_data&BMSR_LSTATUS)) { // Link Down
        if (netif_carrier_ok(netdev)) { // old link state: Up
            printk(KERN_INFO
                   "%s: %s NIC Link is Down\n",
                    at_driver_name,
                    netdev->name );
            adapter->link_speed = SPEED_0;
            clear_bit(0, &adapter->hw.force_ps);
            netif_carrier_off(netdev);
            netif_stop_queue(netdev);
        }
    }
    schedule_work(&adapter->link_chg_task);
}

static inline void
at_clear_phy_int(struct at_adapter* adapter)
{
    u16 phy_data;
    
    spin_lock(&adapter->stats_lock);
    at_read_phy_reg(&adapter->hw, 19, &phy_data);
    spin_unlock(&adapter->stats_lock);
}



/**
 * at_intr - Interrupt Handler
 * @irq: interrupt number
 * @data: pointer to a network interface device structure
 * @pt_regs: CPU registers structure
 **/

static irqreturn_t
at_intr(int irq, void *data)
{
    struct at_adapter *adapter = netdev_priv(data);
    struct at_hw *hw = &adapter->hw;
    u32 status;
    

#ifndef CONFIG_AT_NAPI
    int max_ints = 12;
#endif
    
    //DEBUGFUNC("at_intr !");

    if (0 == (status = adapter->cmb.cmb->int_stats))
        return IRQ_NONE;
        
#ifndef CONFIG_AT_NAPI 
loopint:
#endif
       
    // clear CMB interrupt status at once
    adapter->cmb.cmb->int_stats = 0;

    if (status & ISR_GPHY) { // clear phy status
        at_clear_phy_int(adapter);
    }   
    
    // Ack ISR
    AT_WRITE_REG(hw, REG_ISR, (status|ISR_DIS_INT|ISR_DIS_DMA));  
      
    // check if PCIE PHY Link down
   if (status&ISR_PHY_LINKDOWN) {
        DEBUGOUT1("pcie phy linkdown %x", status);
        if(netif_running(adapter->netdev)) { // reset MAC
            AT_WRITE_REG(hw, REG_ISR, 0);
            AT_WRITE_REG(hw, REG_IMR, 0);
            AT_WRITE_FLUSH(hw);
            schedule_work(&adapter->reset_task);
            return IRQ_HANDLED; 
        }
    }
    
    // check if DMA read/write error ?
    if (status&(ISR_DMAR_TO_RST|ISR_DMAW_TO_RST)) 
    {
        DEBUGOUT1("PCIE DMA RW error (status = 0x%x) !", status);
        //AT_WRITE_REG(&adapter->hw, REG_MASTER_CTRL, MASTER_CTRL_SOFT_RST);
        AT_WRITE_REG(hw, REG_ISR, 0);
        AT_WRITE_REG(hw, REG_IMR, 0);
        AT_WRITE_FLUSH(hw);
        schedule_work(&adapter->reset_task);
        return IRQ_HANDLED; 
    }
             
    // check if SMB intr
    if (status & ISR_SMB) {
        at_inc_smb(adapter);
    }
    

    // link event
    if (status&(ISR_GPHY|ISR_MANUAL))
    {
        //AT_DBG("int-status=0x%x", status);
        adapter->net_stats.tx_carrier_errors++;
        at_check_for_link(adapter);
    }
    
    // transmit event
    if ( status & ISR_TX_EVENT ) {
        at_clean_tx_irq(adapter);
    }
 
    
#ifdef CONFIG_AT_NAPI
    if ( status & ISR_RX_EVENT ) {
        
        struct at_rrd_ring *rrd_ring = &adapter->rrd_ring; 
        rx_return_desc_t* rrd;
        rrd = AT_RRD_DESC(rrd_ring, ((u16)atomic_read(&rrd_ring->next_to_clean)));
        if (rrd->xsz.valid) { // packet valid

            /* disable interrupts, without the synchronize_irq bit */
            atomic_inc(&adapter->irq_sem);
            AT_WRITE_REG(hw, REG_IMR, IMR_NONRX_MASK);
            AT_WRITE_FLUSH(hw);
        
            if (likely(netif_rx_schedule_prep(&adapter->polling_netdev[0])))
                __netif_rx_schedule(&adapter->polling_netdev[0]);
            else
                at_irq_enable(adapter);
        }
    }
 
#else /* if !CONFIG_AT_NAPI */

    
    // recv event
    if ( status & ISR_RX_EVENT ) {
        at_clean_rx_irq(adapter);
    }
 
    if (--max_ints > 0 && 
        0 != (status = adapter->cmb.cmb->int_stats)) {
        goto  loopint;
    }


#endif /* CONFIG_AT_NAPI */

    // re-enable Interrupt
    AT_WRITE_REG(&adapter->hw, REG_ISR, ISR_DIS_SMB|ISR_DIS_DMA);
 
    
    return IRQ_HANDLED;
}
    

static inline u16 tpd_avail(struct at_tpd_ring *tpd_ring)
{
    u16 next_to_clean = (u16)atomic_read(&tpd_ring->next_to_clean);
    u16 next_to_use = (u16)atomic_read(&tpd_ring->next_to_use);
    return ((next_to_clean > next_to_use) ? 
            next_to_clean - next_to_use - 1 :
            tpd_ring->count + next_to_clean - next_to_use - 1);
}

static inline void
at_update_mailbox(struct at_adapter* adapter)
{
    unsigned long flags;
    u32 tpd_next_to_use;
    u32 rfd_next_to_use;
    u32 rrd_next_to_clean;
    
    spin_lock_irqsave(&adapter->mb_lock, flags);
    
    tpd_next_to_use = atomic_read(&adapter->tpd_ring.next_to_use);
    rfd_next_to_use = (u32)atomic_read(&adapter->rfd_ring.next_to_use);
    rrd_next_to_clean = (u32)atomic_read(&adapter->rrd_ring.next_to_clean);
    
    AT_WRITE_REG(&adapter->hw, REG_MAILBOX, 
        ((rfd_next_to_use & MB_RFD_PROD_INDX_MASK) << MB_RFD_PROD_INDX_SHIFT) |
        ((rrd_next_to_clean & MB_RRD_CONS_INDX_MASK) << MB_RRD_CONS_INDX_SHIFT) |
        ((tpd_next_to_use & MB_TPD_PROD_INDX_MASK) << MB_TPD_PROD_INDX_SHIFT) );
        
    spin_unlock_irqrestore(&adapter->mb_lock, flags);
}


static boolean_t
at_clean_tx_irq(struct at_adapter* adapter)
{
    struct at_tpd_ring *tpd_ring = &adapter->tpd_ring; 
    struct at_buffer* buffer_info;
    u16 sw_tpd_next_to_clean;
    u16 cmb_tpd_next_to_clean;

    sw_tpd_next_to_clean = (u16)atomic_read(&tpd_ring->next_to_clean);
    cmb_tpd_next_to_clean = le16_to_cpu(adapter->cmb.cmb->tpd_cons_idx);

    while (cmb_tpd_next_to_clean != sw_tpd_next_to_clean) {
        TpdDescr* tpd;
        
        tpd = AT_TPD_DESC(tpd_ring, sw_tpd_next_to_clean);
        buffer_info = &tpd_ring->buffer_info[sw_tpd_next_to_clean];
        if(buffer_info->dma) {
            pci_unmap_page(adapter->pdev,
                           buffer_info->dma,
                           buffer_info->length,
                           PCI_DMA_TODEVICE);
            buffer_info->dma = 0;
        }

        if(buffer_info->skb) {
            dev_kfree_skb_irq(buffer_info->skb);
            buffer_info->skb = NULL;
        }
        
        if (++sw_tpd_next_to_clean == tpd_ring->count)  
            sw_tpd_next_to_clean = 0;
    }
    atomic_set(&tpd_ring->next_to_clean, sw_tpd_next_to_clean);
    
    if(netif_queue_stopped(adapter->netdev) && 
       netif_carrier_ok(adapter->netdev))
        netif_wake_queue(adapter->netdev);
        
    return (sw_tpd_next_to_clean == (u16)atomic_read(&tpd_ring->next_to_use)) ? TRUE : FALSE;
}

static inline void
at_rx_checksum(struct at_adapter* adapter, 
        rx_return_desc_t* rrd,
        struct sk_buff* skb)
{
    skb->ip_summed = CHECKSUM_NONE;

    if (rrd->pkt_flg&PACKET_FLAG_ERR) {
        if (rrd->err_flg&
            (ERR_FLAG_CRC|ERR_FLAG_TRUNC|ERR_FLAG_CODE|ERR_FLAG_OV)) {
            adapter->hw_csum_err++;
            DEBUGOUT("checksum error !");
            return;
        }
    }

    // none IPv4 
    if (!(rrd->pkt_flg&PACKET_FLAG_IPV4)) {
        // checksum is invalid
        DEBUGOUT("checksum is invalid because of non-IPv4");
        return;
    }
    
    // IPv4 packet
    
    if (!(rrd->err_flg&(ERR_FLAG_IP_CHKSUM|ERR_FLAG_L4_CHKSUM)) ) {
        skb->ip_summed = CHECKSUM_UNNECESSARY;
        adapter->hw_csum_good++;
        DEBUGOUT("IPv4, checkesum ok");
        return;
    }
    
    return ;
}


static void
#ifdef CONFIG_AT_NAPI
at_clean_rx_irq(struct at_adapter *adapter, int *work_done, int work_to_do)
#else
at_clean_rx_irq(struct at_adapter *adapter)
#endif
{
    struct net_device *netdev = adapter->netdev;
    int i, count;
    u16 length, rrd_next_to_clean;
    struct at_rfd_ring *rfd_ring = &adapter->rfd_ring;
    struct at_rrd_ring *rrd_ring = &adapter->rrd_ring; 
    struct at_buffer * buffer_info;
    rx_return_desc_t* rrd;
    struct sk_buff *skb;
    
    count = 0; 
    
    rrd_next_to_clean = (u16)atomic_read(&rrd_ring->next_to_clean);   
    
    while (1)
    {
        rrd = AT_RRD_DESC(rrd_ring, rrd_next_to_clean);
        i = 1;
        if (rrd->xsz.valid) { // packet valid
chk_rrd:        
            // check rrd status
            if (rrd->num_buf != 1) {
                DEBUGOUT1("RRD NumRfd %d", rrd->num_buf);
                DEBUGOUT1("packet length = %d", rrd->xsz.xsum_sz.pkt_size);
            } else {
                goto rrd_ok;
            }
            
            // rrd seems to be bad
            if (i-- > 0) { // rrd may not be DMAed completely
                DEBUGOUT("RRD may not be DMAed completely");
                usec_delay(1);
                goto chk_rrd;
            }
            // bad rrd
            printk("BAD RRD\n");
        
            // see if update RFD index
            if (rrd->num_buf > 1) {
                u16 num_buf;
                num_buf = 
                    (rrd->xsz.xsum_sz.pkt_size+adapter->rx_buffer_len - 1)/
                        adapter->rx_buffer_len;
                DEBUGOUT1("RRD.buf_index (%d)", rrd->buf_indx);
                if (rrd->num_buf == num_buf) {
                    // clean alloc flag for bad rrd
                    while (rfd_ring->next_to_clean != 
                            (rrd->buf_indx + num_buf) ) {
                        DEBUGOUT1("clear index (%d)", rfd_ring->next_to_clean);
                        rfd_ring->buffer_info[rfd_ring->next_to_clean].alloced = 0;
                        if (++rfd_ring->next_to_clean == rfd_ring->count) {
                            rfd_ring->next_to_clean = 0;
                        } 
                    } // end while      
                } // end if (rrd->num_buf == ...)
            }

            // update rrd
            rrd->xsz.valid = 0;
            if (++rrd_next_to_clean == rrd_ring->count)
                rrd_next_to_clean = 0;
                count++; 
                continue;
        } else { // current rrd still not be updated
            break;
        }

rrd_ok:

    //printk(KERN_INFO "Receving Packet!\n");
    
#ifdef CONFIG_AT_NAPI
        if (*work_done >= work_to_do)
            break;
        (*work_done)++;
 #endif
 
 
        // clean alloc flag for bad rrd
        while (rfd_ring->next_to_clean != rrd->buf_indx) {
            rfd_ring->buffer_info[rfd_ring->next_to_clean].alloced = 0;
            if (++rfd_ring->next_to_clean == rfd_ring->count) {
                rfd_ring->next_to_clean = 0;
            }
        }       
    
        buffer_info = &rfd_ring->buffer_info[rrd->buf_indx];
        if (++rfd_ring->next_to_clean == rfd_ring->count) {
            rfd_ring->next_to_clean = 0;
        }
    

        // update rrd next to clean
        if (++rrd_next_to_clean == rrd_ring->count)
            rrd_next_to_clean = 0;
            
        count++; 
            
        if (rrd->pkt_flg&PACKET_FLAG_ERR) {
            if (rrd->err_flg& 
                (ERR_FLAG_CRC|ERR_FLAG_TRUNC|ERR_FLAG_CODE|ERR_FLAG_OV)) {
                if (0 == (netdev->flags & IFF_PROMISC)) {
                    /* packet error , don't need upstream */
                    buffer_info->alloced = 0;  
                    rrd->xsz.valid = 0;
                    DEBUGOUT1("rrd error flag %x", rrd->err_flg);
                    continue;
                }
            }
        }
    
        /* Good Receive */
    
        pci_unmap_page(adapter->pdev,
                       buffer_info->dma,
                       buffer_info->length,
                       PCI_DMA_FROMDEVICE);
        skb = buffer_info->skb;
        length = le16_to_cpu(rrd->xsz.xsum_sz.pkt_size);
            
        skb_put(skb, length - ETHERNET_FCS_SIZE);

        /* Receive Checksum Offload */
        at_rx_checksum(adapter, rrd, skb); 
        skb->protocol = eth_type_trans(skb, adapter->netdev);
/*
        printk(KERN_INFO"pkt:%02x %02x %02x %02x %02x %02x-%02x\n",
                  skb->data[0], skb->data[1], skb->data[2],
                  skb->data[3], skb->data[4], skb->data[5],
                  skb->data[12]);
*/


#ifdef CONFIG_AT_NAPI

#ifdef NETIF_F_HW_VLAN_TX
        if (adapter->vlgrp && (rrd->pkt_flg&PACKET_FLAG_VLAN_INS)) {
            u16 vlan_tag = 
                    (rrd->vlan_tag>>4) |
                    ((rrd->vlan_tag&7)<<13) |
                    ((rrd->vlan_tag&8)<<9);
            DEBUGOUT1("RX VLAN TAG<RRD>=0x%04x", rrd->vlan_tag);
            vlan_hwaccel_receive_skb(skb, adapter->vlgrp, vlan_tag);
        } else 
#endif          
        {
             netif_receive_skb(skb);
        }

#else /* CONFIG_AT_NAPI */

#ifdef NETIF_F_HW_VLAN_TX
        if (adapter->vlgrp && (rrd->pkt_flg&PACKET_FLAG_VLAN_INS)) {
            u16 vlan_tag = 
                    (rrd->vlan_tag>>4) |
                    ((rrd->vlan_tag&7)<<13) |
                    ((rrd->vlan_tag&8)<<9);
            DEBUGOUT1("RX VLAN TAG<RRD>=0x%04x", rrd->vlan_tag);
            vlan_hwaccel_rx(skb, adapter->vlgrp, vlan_tag);
        } else
#endif          
        {    
            netif_rx(skb);
        }

#endif /* CONFIG_AT_NAPI */

        // let protocol layer free skb
        buffer_info->skb = NULL;  
        buffer_info->alloced = 0;  
        rrd->xsz.valid = 0;

        adapter->netdev->last_rx = jiffies;
    }
    
    atomic_set(&rrd_ring->next_to_clean, rrd_next_to_clean);
   
    at_alloc_rx_buffers(adapter);
 
    // update mailbox ?
    if (0 != count) {
        at_update_mailbox(adapter);
    }    
}


static int
at_xmit_frame(struct sk_buff *skb, struct net_device *netdev)
{
    struct at_adapter *adapter = netdev_priv(netdev);
    unsigned long flags;
    unsigned int len = skb->len;
    unsigned int nr_frags = 0;
    int tso;
    u16 tpd_req = 1;
    TpdDescr* pTpd;
    
#ifdef NETIF_F_TSO
    unsigned int mss = 0;
#endif
#ifdef MAX_SKB_FRAGS
    unsigned int f;
    len -= skb->data_len;
#endif    
    
    DEBUGFUNC("at_xmit_frame");    
    
    if (test_bit(__AT_DOWN, &adapter->flags)) {
        dev_kfree_skb_any(skb);
        return NETDEV_TX_OK;
    }
    
    if (unlikely(skb->len <= 0)) {
        dev_kfree_skb_any(skb);
        return NETDEV_TX_OK;
    }
//printk(KERN_INFO "xmit-len=%d\n", skb->len);

#ifdef MAX_SKB_FRAGS
    nr_frags = skb_shinfo(skb)->nr_frags;
    for(f = 0; f < nr_frags; f++) {
        u16 fg_size = skb_shinfo(skb)->frags[f].size;
        if (fg_size) {
            tpd_req += (fg_size + MAX_TX_BUF_LEN - 1) / MAX_TX_BUF_LEN;
        }
    }
#endif//MAX_SKB_FRAGS

#ifdef NETIF_F_TSO
    mss = skb_shinfo(skb)->gso_size;

    if(mss) {
        u8 proto_hdr_len;
        if(skb->protocol == ntohs(ETH_P_IP) 
#ifdef NETIF_F_TSO6            
            || (skb_shinfo(skb)->gso_type == SKB_GSO_TCPV6)
#endif
                                                            ) {         
            proto_hdr_len = skb_transport_offset(skb) + tcp_hdrlen(skb);
            // need additional TPD ?
            if (proto_hdr_len != len) {
                tpd_req += (len - proto_hdr_len + MAX_TX_BUF_LEN - 1) / MAX_TX_BUF_LEN;
            }
        }
    }
#endif//NETIF_F_TSO 

#ifdef NETIF_F_LLTX
    local_irq_save(flags);
    if (!spin_trylock(&adapter->tx_lock)) {
        /* Collision - tell upper layer to requeue */
        local_irq_restore(flags);
        return NETDEV_TX_LOCKED;
    }
#else
    spin_lock_irqsave(&adapter->tx_lock, flags);
#endif

    if(tpd_avail(&adapter->tpd_ring) < tpd_req) {
        // no enough descriptor
        netif_stop_queue(netdev);
        spin_unlock_irqrestore(&adapter->tx_lock, flags);
        return NETDEV_TX_BUSY;
    }
    
    // init tpd flags
    { 
      struct at_tpd_ring* tpd_ring = &adapter->tpd_ring;
      pTpd = AT_TPD_DESC(tpd_ring,
               ((u16)atomic_read(&tpd_ring->next_to_use)));
      memset(pTpd, 0, sizeof(TpdDescr));
    }
    
    
#ifdef NETIF_F_HW_VLAN_TX
    if(unlikely(adapter->vlgrp && vlan_tx_tag_present(skb))) {
            u16 vlan_tag = vlan_tx_tag_get(skb);
            vlan_tag = (vlan_tag << 4)          |
                       (vlan_tag >> 13)         |
                       ((vlan_tag >>9) & 0x8)   ;
        pTpd->ins_vlan = 1;
        pTpd->vlan = vlan_tag;
        DEBUGOUT1("TX VLAN TAG<TPD>=%04x", vlan_tag);
    }
#endif

    if (skb->protocol == ntohs(ETH_P_8021Q))
        pTpd->vlan_tag = 1; // may double-vlan

    // do TSO ?
    tso = at_tso(adapter, skb, pTpd);
    
    if (tso < 0) {
        spin_unlock_irqrestore(&adapter->tx_lock, flags);       
        dev_kfree_skb_any(skb);
        return NETDEV_TX_OK; 
    }

    // do HW-CSUM ?
    if (0 == tso) {
        if (at_tx_csum(adapter, skb, pTpd) < 0) {
            spin_unlock_irqrestore(&adapter->tx_lock, flags);
            dev_kfree_skb_any(skb);
            return NETDEV_TX_OK;
        }
    }
    
    at_tx_map(adapter, skb, pTpd);
    at_tx_queue(adapter, tpd_req, pTpd); 
    
    // update mailbox
    at_update_mailbox(adapter);
    
    spin_unlock_irqrestore(&adapter->tx_lock, flags);
    
    netdev->trans_start = jiffies;
    
    return NETDEV_TX_OK;
}
    

/*
 * at_tso
 * return :
 *  0    -- no tso
 *  1    -- no tso, but tcp/ipv4 xsum
 *  2    -- no tso, but tcp/ipv6 xsum
 *  3    -- do ipv4 tso
 *  4    -- do ipv6 tso
 *  -x   -- bad packet, discard it
 */
static int
at_tso(struct at_adapter* adapter, struct sk_buff *skb, TpdDescr* pTpd)
{
#ifdef NETIF_F_TSO
    u8 hdr_len, ip_off;
    u32 real_len;
    
    int err;

    if (skb_is_gso(skb)) {
        if(skb_header_cloned(skb)) {
            err = pskb_expand_head(skb, 0, 0, GFP_ATOMIC);
            if (err)
                return -1;
        }
        
        if(skb->protocol == htons(ETH_P_IP)) {
            
            real_len = (((unsigned char*)ip_hdr(skb) - skb->data) 
                    + ntohs(ip_hdr(skb)->tot_len));
            if (real_len < skb->len) {
                pskb_trim(skb, real_len);
            }
            
            hdr_len = (skb_transport_offset(skb) + tcp_hdrlen(skb));
            if (skb->len == hdr_len) { // only xsum need
                ip_hdr(skb)->check = 0;
                tcp_hdr(skb)->check = ~csum_tcpudp_magic(
                                        ip_hdr(skb)->saddr,
                                        ip_hdr(skb)->daddr,
                                        tcp_hdrlen(skb),
                                        IPPROTO_TCP,
                                        0);
                pTpd->iphdrlen = ip_hdr(skb)->ihl;
                pTpd->tcphdrlen = tcp_hdrlen(skb) >> 2;
                pTpd->ipxsum = 1;
                pTpd->tcpxsum = 1;
                return 1;
            }
            ip_hdr(skb)->check = 0;
            tcp_hdr(skb)->check = ~csum_tcpudp_magic(
                                        ip_hdr(skb)->saddr,
                                        ip_hdr(skb)->daddr,
                                        0,
                                        IPPROTO_TCP,
                                        0);
            ip_off = (unsigned char*)ip_hdr(skb) - (unsigned char*)skb_network_header(skb);
            if (ip_off == 8) // 802.3-SNAP Header
                pTpd->eth_type = 1;
            else if (ip_off != 0)
                return -2; // bad parameter
            // else // ethernet II
            
            pTpd->iphdrlen = ip_hdr(skb)->ihl;
            pTpd->tcphdrlen = tcp_hdrlen(skb) >> 2; // note L1 driver is wrong here !!
            pTpd->tcp_mss = skb_shinfo(skb)->gso_size;
            pTpd->segment = 1;
           
            return 3;
        }
    }
#endif//NETIF_F_TSO 
    
    return 0;
}


/*
 * at_tx_csum
 * return:
 *  >=0 : OK
 * -x   : failed, discard it
 */
static int
at_tx_csum(struct at_adapter *adapter, struct sk_buff *skb, TpdDescr* pTpd)
{
    TpdCXsumDescr* px = (TpdCXsumDescr*) pTpd;
    u8 css, cso;

    if(skb->ip_summed == CHECKSUM_PARTIAL) {
         
        cso = skb_transport_offset(skb);
        css = cso + skb->csum_offset;
        if(cso&0x1) {
            AT_ERR("payload offset != even number !\n");
            return  -1;
        }
        px->payld_ofs = cso;
        px->psdxsum_ofs = css;
        px->my_xsum = 1;
        DEBUGOUT1("hardware chekcsum: css=%d, cso=%d", css, cso);
    }
    return 0;
}



static void
at_tx_map(struct at_adapter *adapter, struct sk_buff *skb, TpdDescr* pTpd)
{
    struct at_tpd_ring* tpd_ring = &adapter->tpd_ring;
    struct at_buffer *buffer_info;
    u16 buf_len = skb->len;
    struct page *page;                                                       
    unsigned long offset;  
    u16 next_to_use;
    
#ifdef MAX_SKB_FRAGS
    unsigned int nr_frags;
    unsigned int f;
    
    buf_len -= skb->data_len;
    nr_frags = skb_shinfo(skb)->nr_frags;
#endif//MAX_SKB_FRAGS

    
    next_to_use = (u16)atomic_read(&tpd_ring->next_to_use);
    
    buffer_info = tpd_ring->buffer_info+next_to_use; 
    if (buffer_info->skb) {
        BUG();
    }
    buffer_info->skb = NULL;    // put skb in last TPD

#ifdef NETIF_F_TSO
    if (pTpd->segment) { // TSO
        u8 hdr_len = skb_transport_offset(skb) + tcp_hdrlen(skb);
        buffer_info->length = hdr_len;
        page = virt_to_page(skb->data);                                            
        offset = (unsigned long)skb->data & ~PAGE_MASK;                           
        buffer_info->dma =
                        pci_map_page(adapter->pdev,
                                     page,
                                     offset,
                                     hdr_len,
                                     PCI_DMA_TODEVICE);
                                     
        if(++next_to_use == tpd_ring->count) next_to_use = 0;
        
        if (buf_len > hdr_len) {
            u16 data_len = buf_len - hdr_len;
            u16 i, nSeg = (data_len + MAX_TX_BUF_LEN - 1) / MAX_TX_BUF_LEN;
            for (i=0; i < nSeg; i++) {
                buffer_info = tpd_ring->buffer_info+next_to_use; 
                buffer_info->skb = NULL;
                buffer_info->length = 
                    (MAX_TX_BUF_LEN >= data_len) ? MAX_TX_BUF_LEN : data_len;
                data_len -= buffer_info->length;
                page = virt_to_page(skb->data+(hdr_len+i*MAX_TX_BUF_LEN));                                            
                offset = (unsigned long)(skb->data+(i*MAX_TX_BUF_LEN+hdr_len)) & ~PAGE_MASK;
                buffer_info->dma = 
                        pci_map_page(adapter->pdev,
                                     page, 
                                     offset,
                                     buffer_info->length,
                                     PCI_DMA_TODEVICE);
                if(++next_to_use == tpd_ring->count) next_to_use = 0;  
            } 
        }
    } else {
#endif//NETIF_F_TSO
    
    buffer_info->length = buf_len;
    page = virt_to_page(skb->data);                                            
    offset = (unsigned long)skb->data & ~PAGE_MASK;
    buffer_info->dma =
                    pci_map_page(adapter->pdev,
                                 page,
                                 offset,
                                 buf_len,
                                 PCI_DMA_TODEVICE);
    if(++next_to_use == tpd_ring->count) next_to_use = 0;   
#ifdef NETIF_F_TSO
    }    
#endif//NETIF_F_TSO

#ifdef MAX_SKB_FRAGS
    for(f = 0; f < nr_frags; f++) {
        struct skb_frag_struct *frag;
        u16 i, nSeg;
        
        frag = &skb_shinfo(skb)->frags[f];
        buf_len = frag->size;
        
    nSeg = (buf_len + MAX_TX_BUF_LEN - 1) / MAX_TX_BUF_LEN;
        for (i=0; i < nSeg; i++) {
            buffer_info = tpd_ring->buffer_info+next_to_use; 
            if (buffer_info->skb) {
                BUG();
            }
            buffer_info->skb = NULL;
            buffer_info->length = 
                (buf_len > MAX_TX_BUF_LEN) ? MAX_TX_BUF_LEN : buf_len;
            buf_len -= buffer_info->length;
            
            buffer_info->dma =
                pci_map_page(adapter->pdev,
                             frag->page,
                             frag->page_offset+i*MAX_TX_BUF_LEN,
                             buffer_info->length,
                             PCI_DMA_TODEVICE);
                             
            if(++next_to_use == tpd_ring->count) next_to_use = 0; 
        } 
    }
#endif//MAX_SKB_FRAGS
    
    // last tpd's buffer-info
    buffer_info->skb = skb;
}


static void
at_tx_queue(struct at_adapter *adapter, u16 count, TpdDescr* pTpd)
{
    struct at_tpd_ring* tpd_ring = &adapter->tpd_ring;
    struct at_buffer * buffer_info;
    TpdDescr* p;
    u16 j;
    u16 next_to_use = (u16)atomic_read(&tpd_ring->next_to_use);
    
    DEBUGOUT1("<at_tx_quue> count=%d, tpd_desc=%08x,%08x,%08x,%08x", 
               count, 
               *((u32*)pTpd + 0),
               *((u32*)pTpd + 1),
               *((u32*)pTpd + 2),
               *((u32*)pTpd + 3));

        
    for(j=0; j < count; j++) {
        buffer_info = tpd_ring->buffer_info+next_to_use; 
        p = AT_TPD_DESC(tpd_ring, next_to_use);
        if (p != pTpd) {
            memcpy(p, pTpd, sizeof(TpdDescr));
        }         
        p->addr = cpu_to_le64(buffer_info->dma);
        p->buf_len = cpu_to_le16(buffer_info->length);

       /* 
    DEBUGOUT1("<at_tx_quue> TPD_NEXT_TO_USE: %d buf_len=%d", 
            tpd_next_to_use,
            buffer_info->length);
        */
#ifdef NETIF_F_TSO
        p->hdr_flag = ((p->segment)&&(0==j)) ? 1 : 0;
#endif//NETIF_F_TSO

        if (j == (count-1))
            p->eop = 1;
       
        DEBUGOUT1("tpd[%d]=%08x,%08x,%08x,%08x", 
               j, 
               *((u32*)p + 0),
               *((u32*)p + 1),
               *((u32*)p + 2),
               *((u32*)p + 3));

        if(++next_to_use == tpd_ring->count) 
            next_to_use = 0;
    }

    /* Force memory writes to complete before letting h/w
     * know there are new descriptors to fetch.  (Only
     * applicable for weak-ordered memory model archs,
     * such as IA-64). */
    wmb();
   
    atomic_set(&tpd_ring->next_to_use, next_to_use);
}


static void at_inc_smb(struct at_adapter * adapter)
{
    stats_msg_block_t* smb = adapter->smb.smb;
    
    /* Fill out the OS statistics structure */
    adapter->soft_stats.rx_packets          += smb->rx_ok;
    adapter->soft_stats.tx_packets          += smb->tx_ok;
    adapter->soft_stats.rx_bytes            += smb->rx_byte_cnt;
    adapter->soft_stats.tx_bytes            += smb->tx_byte_cnt;
    adapter->soft_stats.multicast           += smb->rx_mcast;
    adapter->soft_stats.collisions          += (smb->tx_1_col + 
                                                smb->tx_2_col*2 + 
                                                smb->tx_late_col + 
                                                smb->tx_abort_col * adapter->hw.max_retry);

    /* Rx Errors */

    adapter->soft_stats.rx_errors           += (smb->rx_frag +
                                                smb->rx_fcs_err + 
                                                smb->rx_len_err + 
                                                smb->rx_sz_ov + 
                                                smb->rx_rxf_ov + 
                                                smb->rx_rrd_ov +
                                                smb->rx_align_err );
    adapter->soft_stats.rx_fifo_errors      += smb->rx_rxf_ov;
    adapter->soft_stats.rx_length_errors    += smb->rx_len_err;
    adapter->soft_stats.rx_crc_errors       += smb->rx_fcs_err;
    adapter->soft_stats.rx_frame_errors     += smb->rx_align_err;
    adapter->soft_stats.rx_missed_errors    += (smb->rx_rrd_ov + 
                                                smb->rx_rxf_ov);
                                                
    adapter->soft_stats.rx_pause            += smb->rx_pause;
    adapter->soft_stats.rx_rrd_ov           += smb->rx_rrd_ov; 
    adapter->soft_stats.rx_trunc            += smb->rx_sz_ov;                                               

    /* Tx Errors */

    adapter->soft_stats.tx_errors           += (smb->tx_late_col +
                                                smb->tx_abort_col +
                                                smb->tx_underrun + 
                                                smb->tx_trunc );
    adapter->soft_stats.tx_fifo_errors      += smb->tx_underrun;
    adapter->soft_stats.tx_aborted_errors   += smb->tx_abort_col;
    adapter->soft_stats.tx_window_errors    += smb->tx_late_col;
    
    adapter->soft_stats.excecol             += smb->tx_abort_col;
    adapter->soft_stats.deffer              += smb->tx_defer;
    adapter->soft_stats.scc                 += smb->tx_1_col;
    adapter->soft_stats.mcc                 += smb->tx_2_col;
    adapter->soft_stats.latecol             += smb->tx_late_col;
    adapter->soft_stats.tx_underun          += smb->tx_underrun;
    adapter->soft_stats.tx_trunc            += smb->tx_trunc;
    adapter->soft_stats.tx_pause            += smb->tx_pause;


    // finally
        
    adapter->net_stats.rx_packets           = adapter->soft_stats.rx_packets;                  
    adapter->net_stats.tx_packets           = adapter->soft_stats.tx_packets;                     
    adapter->net_stats.rx_bytes             = adapter->soft_stats.rx_bytes;                     
    adapter->net_stats.tx_bytes             = adapter->soft_stats.tx_bytes;            
    adapter->net_stats.multicast            = adapter->soft_stats.multicast;           
    adapter->net_stats.collisions           = adapter->soft_stats.collisions;          
    adapter->net_stats.rx_errors            = adapter->soft_stats.rx_errors;          
    adapter->net_stats.rx_over_errors       = adapter->soft_stats.rx_missed_errors;
    adapter->net_stats.rx_length_errors     = adapter->soft_stats.rx_length_errors;    
    adapter->net_stats.rx_crc_errors        = adapter->soft_stats.rx_crc_errors;       
    adapter->net_stats.rx_frame_errors      = adapter->soft_stats.rx_frame_errors;     
    adapter->net_stats.rx_fifo_errors       = adapter->soft_stats.rx_fifo_errors;      
    adapter->net_stats.rx_missed_errors     = adapter->soft_stats.rx_missed_errors;    
    adapter->net_stats.tx_errors            = adapter->soft_stats.tx_errors;           
    adapter->net_stats.tx_fifo_errors       = adapter->soft_stats.tx_fifo_errors;      
    adapter->net_stats.tx_aborted_errors    = adapter->soft_stats.tx_aborted_errors;  
    adapter->net_stats.tx_window_errors     = adapter->soft_stats.tx_window_errors;    
    adapter->net_stats.tx_carrier_errors    = adapter->soft_stats.tx_carrier_errors;  
}


/**
 * at_phy_config - Timer Call-back
 * @data: pointer to netdev cast into an unsigned long
 **/

static void
at_phy_config(unsigned long data)
{
    struct at_adapter *adapter = (struct at_adapter *) data;
    struct at_hw *hw = &adapter->hw;  
    unsigned long flags;

     DEBUGFUNC("at_phy_reconfig!");
    
    spin_lock_irqsave(&adapter->stats_lock, flags);
    at_write_phy_reg(hw, MII_ADVERTISE, hw->mii_autoneg_adv_reg);
    at_write_phy_reg(hw, MII_AT001_CR, hw->mii_1000t_ctrl_reg);
    
    DEBUGOUT("4 register written");
    at_write_phy_reg(hw, MII_BMCR, MII_CR_RESET|MII_CR_AUTO_NEG_EN|MII_CR_RESTART_AUTO_NEG);
    spin_unlock_irqrestore(&adapter->stats_lock, flags);
    clear_bit(0, &adapter->cfg_phy);
}


/**
 * at_watchdog - Timer Call-back
 * @data: pointer to netdev cast into an unsigned long
 **/

static void
at_watchdog(unsigned long data)
{
    struct at_adapter *adapter = (struct at_adapter *) data;
    unsigned long flags;
    
    spin_lock_irqsave(&adapter->stats_lock, flags);

    spin_unlock_irqrestore(&adapter->stats_lock, flags);
    
/*        
    if (!netif_carrier_ok(adapter->netdev)) {
        schedule_work(&adapter->link_chg_task);
    }
*/

    /* Reset the timer */
    mod_timer(&adapter->watchdog_timer, jiffies + 4 * HZ);
}




