#include "at.h"

/* This is the only thing that needs to be changed to adjust the
 * maximum number of ports that the driver can manage.
 */

#define AT_MAX_NIC 32

#define OPTION_UNSET    -1
#define OPTION_DISABLED 0
#define OPTION_ENABLED  1



 /* All parameters are treated the same, as an integer array of values.
 * This macro just reduces the need to repeat the same declaration code
 * over and over (plus this helps to avoid typo bugs).
 */
#define AT_PARAM_INIT { [0 ... AT_MAX_NIC] = OPTION_UNSET }
#ifndef module_param_array
/* Module Parameters are always initialized to -1, so that the driver
 * can tell the difference between no user specified value or the
 * user asking for the default value.
 * The true default values are loaded in when e1000_check_options is called.
 *
 * This is a GCC extension to ANSI C.
 * See the item "Labeled Elements in Initializers" in the section
 * "Extensions to the C Language Family" of the GCC documentation.
 */

#define AT_PARAM(X, desc) \
	static const int __devinitdata X[AT_MAX_NIC+1] = AT_PARAM_INIT; \
	MODULE_PARM(X, "1-" __MODULE_STRING(AT_MAX_NIC) "i"); \
	MODULE_PARM_DESC(X, desc);
#else
#define AT_PARAM(X, desc) \
	static int __devinitdata X[AT_MAX_NIC+1] = AT_PARAM_INIT; \
	static int num_##X = 0; \
	module_param_array_named(X, X, int, &num_##X, 0); \
	MODULE_PARM_DESC(X, desc);
#endif

/* Transmit Descriptor Count
 *
 * Valid Range: 64-2048 
 *
 * Default Value: 128
 */

AT_PARAM(TxDescriptors, "Number of transmit descriptors");

/* Receive Descriptor Count
 *
 * Valid Range: 64-4096
 *
 * Default Value: 256
 */

AT_PARAM(RxDescriptors, "Number of receive descriptors");

/* User Specified MediaType Override
 *
 * Valid Range: 0-5
 *  - 0    - auto-negotiate at all supported speeds
 *  - 1    - only link at 1000Mbps Full Duplex
 *  - 2    - only link at 100Mbps Full Duplex
 *  - 3    - only link at 100Mbps Half Duplex
 *  - 4    - only link at 10Mbps Full Duplex
 *  - 5    - only link at 10Mbps Half Duplex
 * Default Value: 0
 */

AT_PARAM(MediaType, "MediaType Select");

/* Interrupt Moderate Timer in units of 2 us
 *
 * Valid Range: 10-65535
 *
 * Default Value: 45000(90ms)
 */

AT_PARAM(IntModTimer, "Interrupt Moderator Timer");



/* FlashVendor
 * Valid Range: 0-2
 * 0 - Atmel
 * 1 - SST
 * 2 - ST
 */

AT_PARAM(FlashVendor, "SPI Flash Vendor");



#define AUTONEG_ADV_DEFAULT  0x2F
#define AUTONEG_ADV_MASK     0x2F
#define FLOW_CONTROL_DEFAULT FLOW_CONTROL_FULL

#define DEFAULT_INT_MOD_CNT             100 // 200us
#define MAX_INT_MOD_CNT                 65000
#define MIN_INT_MOD_CNT                 50

#define FLASH_VENDOR_DEFAULT    0
#define FLASH_VENDOR_MIN        0
#define FLASH_VENDOR_MAX        2


struct at_option {
	enum { enable_option, range_option, list_option } type;
	char *name;
	char *err;
	int  def;
	union {
		struct { /* range_option info */
			int min;
			int max;
		} r;
		struct { /* list_option info */
			int nr;
			struct at_opt_list { int i; char *str; } *p;
		} l;
	} arg;
};

static int __devinit
at_validate_option(int *value, struct at_option *opt)
{
	if(*value == OPTION_UNSET) {
		*value = opt->def;
		return 0;
	}

	switch (opt->type) {
	case enable_option:
		switch (*value) {
		case OPTION_ENABLED:
			printk(KERN_INFO "%s Enabled\n", opt->name);
			return 0;
		case OPTION_DISABLED:
			printk(KERN_INFO "%s Disabled\n", opt->name);
			return 0;
		}
		break;
	case range_option:
		if(*value >= opt->arg.r.min && *value <= opt->arg.r.max) {
			printk(KERN_INFO "%s set to %i\n", opt->name, *value);
			return 0;
		}
		break;
	case list_option: {
		int i;
		struct at_opt_list *ent;

		for(i = 0; i < opt->arg.l.nr; i++) {
			ent = &opt->arg.l.p[i];
			if(*value == ent->i) {
				if(ent->str[0] != '\0')
					printk(KERN_INFO "%s\n", ent->str);
				return 0;
			}
		}
	}
		break;
	default:
		BUG();
	}

	printk(KERN_INFO "Invalid %s specified (%i) %s\n",
	       opt->name, *value, opt->err);
	*value = opt->def;
	return -1;
}

/**
 * at_check_options - Range Checking for Command Line Parameters
 * @adapter: board private structure
 *
 * This routine checks all command line parameters for valid user
 * input.  If an invalid value is given, or if no user specified
 * value exists, a default value is used.  The final value is stored
 * in a variable in the adapter structure.
 **/

void __devinit
at_check_options(struct at_adapter *adapter)
{
	int bd = adapter->bd_number;
	if(bd >= AT_MAX_NIC) {
		printk(KERN_NOTICE
		       "Warning: no configuration for board #%i\n", bd);
		printk(KERN_NOTICE "Using defaults for all values\n");
#ifndef module_param_array
		bd = AT_MAX_NIC;
#endif
	}

	{ /* Transmit Descriptor Count */
		struct at_option opt = {
			.type = range_option,
			.name = "Transmit Descriptors",
			.err  = "using default of "
				__MODULE_STRING(AT_DEFAULT_TPD),
			.def  = AT_DEFAULT_TPD,
			.arg  = { .r = { .min = AT_MIN_TPD, .max = AT_MAX_TPD }}
		};
		struct at_tpd_ring *tpd_ring = &adapter->tpd_ring;
		int val;
#ifdef module_param_array
		if(num_TxDescriptors > bd) {
#endif
		    val = TxDescriptors[bd];
		    at_validate_option(&val, &opt);
		    tpd_ring->count = ((uint16_t) val)&~1;
#ifdef module_param_array
		} else {
			tpd_ring->count = (uint16_t)opt.def;
		}
#endif
	}
	
	{ /* Receive Descriptor Count */
		struct at_option opt = {
			.type = range_option,
			.name = "Receive Descriptors",
			.err  = "using default of "
				__MODULE_STRING(AT_DEFAULT_RFD),
			.def  = AT_DEFAULT_RFD,
			.arg  = { .r = { .min = AT_MIN_RFD, .max = AT_MAX_RFD }}
		};
		struct at_rfd_ring *rfd_ring = &adapter->rfd_ring;
        	struct at_rrd_ring * rrd_ring = &adapter->rrd_ring;
		int val;
#ifdef module_param_array
		if(num_RxDescriptors > bd) {
#endif			
        	val = RxDescriptors[bd];
		    at_validate_option(&val, &opt);
		    rrd_ring->count = rfd_ring->count = ((uint16_t)val)&~1; // even number
#ifdef module_param_array
		} else {
			rfd_ring->count = rrd_ring->count = (uint16_t)opt.def;
		}
#endif
	}
	
	{ /* Interrupt Moderate Timer */
	    struct at_option opt = { 
	        .type = range_option,
	        .name = "Interrupt Moderate Timer",
	        .err  = "using default of " __MODULE_STRING(DEFAULT_INT_MOD_CNT),
	        .def  = DEFAULT_INT_MOD_CNT,
	        .arg  = { .r = { .min = MIN_INT_MOD_CNT, .max = MAX_INT_MOD_CNT }}
	    } ;
        int val;
#ifdef module_param_array
		if(num_IntModTimer > bd) {
#endif	        
        val = IntModTimer[bd];
	    at_validate_option(&val, &opt);	
            adapter->imt = (uint16_t) val;   
#ifdef module_param_array
		} else {
			adapter->imt = (uint16_t)(opt.def);
		}
#endif	             
	}
	
	{ /* Flsh Vendor */
	    struct at_option opt = { 
	        .type = range_option,
	        .name = "SPI Flash Vendor",
	        .err  = "using default of " __MODULE_STRING(FLASH_VENDOR_DEFAULT),
	        .def  = DEFAULT_INT_MOD_CNT,
	        .arg  = { .r = { .min = FLASH_VENDOR_MIN, .max = FLASH_VENDOR_MAX }}
	    } ;
        int val;
#ifdef module_param_array
		if(num_FlashVendor > bd) {
#endif	        
        val = FlashVendor[bd];
	    at_validate_option(&val, &opt);	
        adapter->hw.flash_vendor = (uint8_t) val;   
#ifdef module_param_array
		} else {
			adapter->hw.flash_vendor = (uint8_t)(opt.def);
		}
#endif	             
	}
	
	{ /* MediaType */
	    struct at_option opt = { 
	        .type = range_option,
	        .name = "Speed/Duplex Selection",
	        .err  = "using default of " __MODULE_STRING(MEDIA_TYPE_AUTO_SENSOR),
	        .def  = MEDIA_TYPE_AUTO_SENSOR,
	        .arg  = { .r = { .min = MEDIA_TYPE_AUTO_SENSOR, .max = MEDIA_TYPE_10M_HALF }}
	    } ;
        int val;
#ifdef module_param_array
		if(num_MediaType > bd) {
#endif	        
        val = MediaType[bd];
	    at_validate_option(&val, &opt);	
	    adapter->hw.MediaType = (uint16_t) val;
#ifdef module_param_array
		} else {
			adapter->hw.MediaType = (uint16_t)(opt.def);
		}
#endif	             
	}
}

