#include "at.h"

#define RUN_REALTIME 0

char at_driver_name[] = "ATL1";
char at_driver_string[] = "Attansic(R) L1 Ethernet Network Driver";
#define DRV_VERSION "1.0.41.0"
char at_driver_version[] = DRV_VERSION;
char at_copyright[] = "Copyright (c) 1999-2005 Attansic Corporation.";

/* 
 * at_pci_tbl - PCI Device ID Table
 *
 * Wildcard entries (PCI_ANY_ID) should come last
 * Last entry must be all 0s
 *
 * { Vendor ID, Device ID, SubVendor ID, SubDevice ID,
 *   Class, Class Mask, private data (not used) }
 */
static struct pci_device_id at_pci_tbl[] = {
    ATTANSIC_ETHERNET_DEVICE(0x1048),
    /* required last entry */
    {0,}
};

MODULE_DEVICE_TABLE(pci, at_pci_tbl);

int32_t at_up(struct at_adapter *adapter);
void at_down(struct at_adapter *adapter);
int at_reset(struct at_adapter *adapter);
int32_t at_setup_ring_resources(struct at_adapter *adapter);
void at_free_ring_resources(struct at_adapter *adapter);

/* Local Function Prototypes */
static int at_init_module(void);
static void at_exit_module(void);
static int at_probe(struct pci_dev *pdev, const struct pci_device_id *ent);
static void __devexit at_remove(struct pci_dev *pdev);
static int at_sw_init(struct at_adapter *adapter);
static int at_open(struct net_device *netdev);
static int at_close(struct net_device *netdev);
static int at_xmit_frame(struct sk_buff *skb, struct net_device *netdev);
static struct net_device_stats * at_get_stats(struct net_device *netdev);
static int at_change_mtu(struct net_device *netdev, int new_mtu);
static void at_set_multi(struct net_device *netdev);
static int at_set_mac(struct net_device *netdev, void *p);
static int at_ioctl(struct net_device *netdev, struct ifreq *ifr, int cmd);
static void at_tx_timeout(struct net_device *dev);
static irqreturn_t at_intr(int irq, void *data, struct pt_regs *regs);
static void at_inc_smb(struct at_adapter * adapter);
static void at_intr_rx(struct at_adapter* adapter);
static void at_intr_tx(struct at_adapter* adapter);

static void at_watchdog(unsigned long data);
static void at_phy_config(unsigned long data);
static void at_tx_timeout_task(struct net_device *netdev);
static void at_check_for_link(struct at_adapter* adapter);
static void at_link_chg_task(struct net_device* netdev);
static uint32_t at_check_link(struct at_adapter* adapter);

static void at_clean_tx_ring(struct at_adapter *adapter);
static void at_clean_rx_ring(struct at_adapter *adapter);
static uint16_t at_alloc_rx_buffers(struct at_adapter *adapter);
static uint32_t at_configure(struct at_adapter *adapter);
static void at_pcie_patch(struct at_adapter *adapter);

#ifdef SIOCGMIIPHY
static int at_mii_ioctl(struct net_device *netdev, struct ifreq *ifr, int cmd);
#endif


#ifdef NETIF_F_HW_VLAN_TX
static void at_vlan_rx_register(struct net_device *netdev, struct vlan_group *grp);
static void at_vlan_rx_add_vid(struct net_device *netdev, uint16_t vid);
static void at_vlan_rx_kill_vid(struct net_device *netdev, uint16_t vid);
static void at_restore_vlan(struct at_adapter *adapter);
#endif

static int at_notify_reboot(struct notifier_block *nb, unsigned long event, void *p);
static int at_suspend(struct pci_dev *pdev, uint32_t state);
#ifdef CONFIG_PM
static int at_resume(struct pci_dev *pdev);
#endif

static void at_via_workaround(struct at_adapter * adapter);


struct notifier_block at_notifier_reboot = {
    .notifier_call  = at_notify_reboot,
    .next       = NULL,
    .priority   = 0
};

/* Exported from other modules */

extern void at_check_options(struct at_adapter *adapter);
extern int at_ethtool_ioctl(struct net_device *netdev, struct ifreq *ifr);
#ifdef SIOCDEVPRIVATE
extern int at_priv_ioctl(struct net_device* netdev, struct ifreq* ifr);
#endif

static struct pci_driver at_driver = {
    .name     = at_driver_name,
    .id_table = at_pci_tbl,
    .probe    = at_probe,
    .remove   = __devexit_p(at_remove),
    /* Power Managment Hooks */
#ifdef CONFIG_PM
    .suspend  = at_suspend,
    .resume   = at_resume
#endif
};

MODULE_AUTHOR("Attansic Corporation, <xiong_huang@attansic.com>");
MODULE_DESCRIPTION("Attansic 1000M Ethernet Network Driver");
MODULE_LICENSE("GPL");
MODULE_VERSION(DRV_VERSION);

/**
 * at_init_module - Driver Registration Routine
 *
 * at_init_module is the first routine called when the driver is
 * loaded. All it does is register with the PCI subsystem.
 **/

static int __init
at_init_module(void)
{
    int ret;
    printk(KERN_INFO "%s - version %s\n",
           at_driver_string, at_driver_version);

    printk(KERN_INFO "%s\n", at_copyright);

    ret = pci_module_init(&at_driver);
    if(ret >= 0) {
        register_reboot_notifier(&at_notifier_reboot);
    }
    return ret;
}

module_init(at_init_module);

/**
 * at_exit_module - Driver Exit Cleanup Routine
 *
 * at_exit_module is called just before the driver is removed
 * from memory.
 **/

static void __exit
at_exit_module(void)
{
    unregister_reboot_notifier(&at_notifier_reboot);
    pci_unregister_driver(&at_driver);
}

module_exit(at_exit_module);


/**
 * at_probe - Device Initialization Routine
 * @pdev: PCI device information struct
 * @ent: entry in at_pci_tbl
 *
 * Returns 0 on success, negative on failure
 *
 * at_probe initializes an adapter identified by a pci_dev structure.
 * The OS initialization, configuring of the adapter private structure,
 * and a hardware reset occur.
 **/

static int __devinit
at_probe(struct pci_dev *pdev,
            const struct pci_device_id *ent)
{
    struct net_device *netdev;
    struct at_adapter *adapter;
    static int cards_found = 0;
    unsigned long mmio_start;
    int mmio_len;
    boolean_t pci_using_64 = TRUE;
    int err;
//    uint16_t eeprom_data;

    DEBUGFUNC("at_probe !");

    if((err = pci_enable_device(pdev)))
        return err;

    if((err = pci_set_dma_mask(pdev, PCI_DMA_64BIT))) {
        if((err = pci_set_dma_mask(pdev, PCI_DMA_32BIT))) {
            AT_ERR("No usable DMA configuration, aborting\n");
            return err;
        }
        pci_using_64 = FALSE;
    }

    
    // Mark all PCI regions associated with PCI device 
    // pdev as being reserved by owner at_driver_name
    if((err = pci_request_regions(pdev, at_driver_name)))
        return err;

    // Enables bus-mastering on the device and calls 
    // pcibios_set_master to do the needed arch specific settings
    pci_set_master(pdev);

    netdev = alloc_etherdev(sizeof(struct at_adapter));
    if(!netdev) {
        err = -ENOMEM;
        goto err_alloc_etherdev;
    }

    SET_MODULE_OWNER(netdev);
    SET_NETDEV_DEV(netdev, &pdev->dev);

    pci_set_drvdata(pdev, netdev);
    adapter = netdev_priv(netdev);
    adapter->netdev = netdev;
    adapter->pdev = pdev;
    adapter->hw.back = adapter;

    mmio_start = pci_resource_start(pdev, BAR_0);
    mmio_len = pci_resource_len(pdev, BAR_0);

    AT_DBG("base memory = %lx memory length = %x \n", 
        mmio_start, mmio_len);
    adapter->hw.mem_rang = (uint32_t)mmio_len;
    adapter->hw.hw_addr = ioremap_nocache(mmio_start, mmio_len);
    if(!adapter->hw.hw_addr) {
        err = -EIO;
        goto err_ioremap;
    }
    /* get device reversion number */
    adapter->hw.dev_rev = 
	    AT_READ_REGW(&adapter->hw, (REG_MASTER_CTRL+2));

    netdev->open = &at_open;
    netdev->stop = &at_close;
    netdev->hard_start_xmit = &at_xmit_frame;
    netdev->get_stats = &at_get_stats;
    netdev->set_multicast_list = &at_set_multi;
    netdev->set_mac_address = &at_set_mac;
    netdev->change_mtu = &at_change_mtu;
    netdev->do_ioctl = &at_ioctl;
#ifdef HAVE_TX_TIMEOUT
    netdev->tx_timeout = &at_tx_timeout;
    netdev->watchdog_timeo = 5 * HZ;
#endif
#ifdef NETIF_F_HW_VLAN_TX
    netdev->vlan_rx_register = at_vlan_rx_register;
    netdev->vlan_rx_add_vid = at_vlan_rx_add_vid;
    netdev->vlan_rx_kill_vid = at_vlan_rx_kill_vid;
#endif

    netdev->mem_start = mmio_start;
    netdev->mem_end = mmio_start + mmio_len;
    //netdev->base_addr = adapter->io_base;
    adapter->bd_number = cards_found;
    adapter->pci_using_64 = pci_using_64;

    /* setup the private structure */

    if((err = at_sw_init(adapter)))
        goto err_sw_init;

    netdev->features = NETIF_F_HW_CSUM;
    
#ifdef MAX_SKB_FRAGS
    netdev->features |= NETIF_F_SG;
#endif
#ifdef NETIF_F_HW_VLAN_TX
    netdev->features |= 
               (NETIF_F_HW_VLAN_TX | 
                NETIF_F_HW_VLAN_RX );
#endif
#ifdef NETIF_F_TSO
    netdev->features |= NETIF_F_TSO;
#endif/*NETIF_F_TSO*/

    if(pci_using_64) {
        netdev->features |= NETIF_F_HIGHDMA;
        AT_DBG("pci using 64bit address\n");
    }
#ifdef NETIF_F_LLTX
	netdev->features |= NETIF_F_LLTX;
#endif

    /* patch for some L1 of old version,
     * the final version of L1 may not need these 
     * patches 
     */
    //KW 
    //at_pcie_patch(adapter);    

    /* really reset GPHY core */
    AT_WRITE_REGW(&adapter->hw, REG_GPHY_ENABLE, 0);

    /* reset the controller to 
     * put the device in a known good starting state */
    
    if (at_reset_hw(&adapter->hw)) {
        err = -EIO;
        goto err_reset;
    }

    /* copy the MAC address out of the EEPROM */

    at_read_mac_addr(&adapter->hw);
    memcpy(netdev->dev_addr, adapter->hw.mac_addr, netdev->addr_len);

    if(!is_valid_ether_addr(netdev->dev_addr)) {
        err = -EIO;
        goto err_eeprom;
    }

    AT_DBG("mac address : %02x-%02x-%02x-%02x-%02x-%02x\n",
        adapter->hw.mac_addr[0],
        adapter->hw.mac_addr[1],
        adapter->hw.mac_addr[2],
        adapter->hw.mac_addr[3],
        adapter->hw.mac_addr[4],
        adapter->hw.mac_addr[5] );

    at_check_options(adapter);
    
    /* pre-init the MAC, and setup link */

    if ((err = at_init_hw(&adapter->hw))) {
        err = -EIO;
        goto err_init_hw;
    }
    
    //KW add
    at_pcie_patch(adapter);
    /* assume we have no link for now */

    netif_carrier_off(netdev);
    netif_stop_queue(netdev);
    
    init_timer(&adapter->watchdog_timer);
    adapter->watchdog_timer.function = &at_watchdog;
    adapter->watchdog_timer.data = (unsigned long) adapter;
    
    init_timer(&adapter->phy_config_timer);
    adapter->phy_config_timer.function = &at_phy_config;
    adapter->phy_config_timer.data = (unsigned long) adapter;
    adapter->phy_timer_pending = FALSE;
    
    INIT_WORK(&adapter->tx_timeout_task,
        (void (*)(void *))at_tx_timeout_task, netdev);
        
    INIT_WORK(&adapter->link_chg_task, 
        (void (*)(void *))at_link_chg_task, netdev);
   
    INIT_WORK(&adapter->pcie_dma_to_rst_task,
        (void (*)(void *))at_tx_timeout_task, netdev);

    if((err = register_netdev(netdev)))
        goto err_register;

    
    cards_found++;

	at_via_workaround(adapter);

    return 0;

err_init_hw:
err_reset:
err_register:
err_sw_init:
err_eeprom:
    iounmap(adapter->hw.hw_addr);
err_ioremap:
    free_netdev(netdev);
err_alloc_etherdev:
    pci_release_regions(pdev);
    return err;
}

/**
 * at_pcie_patch - Patch for PCIE module
 **/
void at_pcie_patch(struct at_adapter* adapter)
{
    uint32_t value;
    value = 0x6500;
    AT_WRITE_REG(&adapter->hw, 0x12FC, value);    

    /* pcie flow control mode change */
    value = AT_READ_REG(&adapter->hw, 0x1008);
    value |= 0x8000;
    AT_WRITE_REG(&adapter->hw, 0x1008, value);
}

/**
 * at_remove - Device Removal Routine
 * @pdev: PCI device information struct
 *
 * at_remove is called by the PCI subsystem to alert the driver
 * that it should release a PCI device.  The could be caused by a
 * Hot-Plug event, or because the driver is going to be removed from
 * memory.
 **/

static void __devexit
at_remove(struct pci_dev *pdev)
{
    struct net_device *netdev = pci_get_drvdata(pdev);
    struct at_adapter *adapter;

    DEBUGFUNC("at_remove");

    /* Device not available. Return. */
    if (!netdev)  
        return;

    adapter = netdev_priv(netdev);
    
    AT_WRITE_REGW(&adapter->hw, REG_GPHY_ENABLE, 0);

    unregister_netdev(netdev);

    iounmap(adapter->hw.hw_addr);
    pci_release_regions(pdev);

    free_netdev(netdev);
}


/**
 * at_irq_disable - Mask off interrupt generation on the NIC
 * @adapter: board private structure
 **/
inline void
at_irq_disable(struct at_adapter *adapter)
{
    atomic_inc(&adapter->irq_sem);
    AT_WRITE_REG(&adapter->hw, REG_IMR, 0);
    
    synchronize_irq(adapter->pdev->irq);
}

/**
 * at_irq_enable - Enable default interrupt generation settings
 * @adapter: board private structure
 **/
inline void
at_irq_enable(struct at_adapter *adapter)
{
    if(0 == atomic_dec_and_test(&adapter->irq_sem)) {
        AT_WRITE_REG(&adapter->hw, REG_IMR, IMR_NORMAL_MASK);
    }
}

static int
at_notify_reboot(struct notifier_block *nb, unsigned long event, void *p)
{
    struct pci_dev *pdev = NULL;

    DEBUGFUNC("at_notify_reboot !");

    switch(event) {
    case SYS_DOWN:
    case SYS_HALT:
    case SYS_POWER_OFF:
        while((pdev = pci_find_device(PCI_ANY_ID, PCI_ANY_ID, pdev))) {
            if(pci_dev_driver(pdev) == &at_driver)
                at_suspend(pdev, 3);
        }
    }
    return NOTIFY_DONE;
}


static int
at_suspend(struct pci_dev *pdev, uint32_t state)
{
    struct net_device *netdev = pci_get_drvdata(pdev);
    struct at_adapter *adapter = netdev_priv(netdev);
    struct at_hw * hw = &adapter->hw;
    uint32_t ctrl = 0;
    uint32_t wufc = adapter->wol;

    DEBUGFUNC("at_suspend !"); 

    netif_device_detach(netdev);

    if(netif_running(netdev))
        at_down(adapter);

    at_read_phy_reg(hw, MII_BMSR, (uint16_t*)&ctrl);
    at_read_phy_reg(hw, MII_BMSR, (uint16_t*)&ctrl);
    if(ctrl & BMSR_LSTATUS)
        wufc &= ~AT_WUFC_LNKC;
        
    // reduce speed to 10/100M
    if (wufc) {
        at_phy_enter_power_saving(hw);
        // if resume, let driver to re- setup link
        hw->phy_configured = FALSE; 
        set_mac_addr(hw);
        at_set_multi(netdev);
        
        ctrl = 0;
        /* turn on magic packet wol */
        if (wufc & AT_WUFC_MAG) {
            ctrl = WOL_MAGIC_EN|WOL_MAGIC_PME_EN;
        }
        // turn on Link change WOL */
        if (wufc & AT_WUFC_LNKC) {
            ctrl |= (WOL_LINK_CHG_EN|WOL_LINK_CHG_PME_EN);
        }
        AT_WRITE_REG(hw, REG_WOL_CTRL, ctrl);
            
        /* turn on all-multi mode if wake on multicast is enabled */
        ctrl = AT_READ_REG(hw, REG_MAC_CTRL);
        ctrl &= ~MAC_CTRL_DBG;
        ctrl &= ~MAC_CTRL_PROMIS_EN;
		if(wufc & AT_WUFC_MC) {
		    ctrl |= MAC_CTRL_MC_ALL_EN;
		} else {
		    ctrl &= ~MAC_CTRL_MC_ALL_EN;
		}
		/* turn on broadcast mode if wake on-BC is enabled */
		if (wufc & AT_WUFC_BC) {
		    ctrl |= MAC_CTRL_BC_EN;
		} else {
		    ctrl &= ~MAC_CTRL_BC_EN;
		}
		
		// enable RX
		ctrl |= MAC_CTRL_RX_EN;
		AT_WRITE_REG(hw, REG_MAC_CTRL, ctrl);
		
		pci_enable_wake(pdev, 3, 1);
		pci_enable_wake(pdev, 4, 1); /* 4 == D3 cold */
	} else {
	    AT_WRITE_REG(hw, REG_WOL_CTRL, 0);
		pci_enable_wake(pdev, 3, 0);
		pci_enable_wake(pdev, 4, 0); /* 4 == D3 cold */
	}
		
	pci_save_state(pdev);
	pci_disable_device(pdev);

    state = (state > 0) ? 3 : 0;
    pci_set_power_state(pdev, state);

    return 0;
}

static int
at_resume(struct pci_dev *pdev)
{
    struct net_device *netdev = pci_get_drvdata(pdev);
    struct at_adapter *adapter = netdev_priv(netdev);
    uint32_t ret_val;

    DEBUGFUNC("at_resume !");

    pci_set_power_state(pdev, 0);
	pci_restore_state(pdev);
	
    ret_val = pci_enable_device(pdev);
    pci_enable_wake(pdev, 3, 0);
    pci_enable_wake(pdev, 4, 0); /* 4 == D3 cold */
    
    AT_WRITE_REG(&adapter->hw, REG_WOL_CTRL, 0);
    at_reset(adapter);
    

    if(netif_running(netdev))
        at_up(adapter);

    netif_device_attach(netdev);

	at_via_workaround(adapter);

    return 0;
}


/**
 * at_sw_init - Initialize general software structures (struct at_adapter)
 * @adapter: board private structure to initialize
 *
 * at_sw_init initializes the Adapter private data structure.
 * Fields are initialized based on PCI device information and
 * OS network device settings (MTU size).
 **/

static int __devinit
at_sw_init(struct at_adapter *adapter)
{
    struct at_hw *hw = &adapter->hw;
    struct net_device *netdev = adapter->netdev;
    struct pci_dev *pdev = adapter->pdev;

    /* PCI config space info */

    hw->vendor_id = pdev->vendor;
    hw->device_id = pdev->device;
    hw->subsystem_vendor_id = pdev->subsystem_vendor;
    hw->subsystem_id = pdev->subsystem_device;

    pci_read_config_byte(pdev, PCI_REVISION_ID, &hw->revision_id);

    pci_read_config_word(pdev, PCI_COMMAND, &hw->pci_cmd_word);


    hw->max_frame_size = netdev->mtu +
                 ENET_HEADER_SIZE + ETHERNET_FCS_SIZE;
    hw->min_frame_size = MINIMUM_ETHERNET_FRAME_SIZE;
    
    adapter->wol = 0;
    
    adapter->rx_buffer_len = (hw->max_frame_size+7)&~7;
    adapter->ict = 50000;  // 100ms
    
    adapter->link_speed = SPEED_0;   // hardware init
    adapter->link_duplex = FULL_DUPLEX; //

  
    hw->phy_configured = FALSE;
    hw->preamble_len = 7;
    hw->ipgt = 0x60;
    hw->min_ifg = 0x50;
    hw->ipgr1 = 0x40;
    hw->ipgr2 = 0x60;
    
    hw->max_retry = 0xf;
    hw->lcol = 0x37;
    hw->jam_ipg = 7;
    
    hw->rfd_burst = 8;
    hw->rrd_burst = 8;
    hw->rfd_fetch_gap = 1;
    hw->rx_jumbo_th = adapter->rx_buffer_len / 8;
    hw->rx_jumbo_lkah = 1;
    hw->rrd_ret_timer = 16;
    
    hw->tpd_burst = 4;
    hw->tpd_fetch_th = 16; 
    hw->txf_burst = 0x100;
    hw->tx_jumbo_task_th = (hw->max_frame_size+7)>>3;
    hw->tpd_fetch_gap = 1;
    
    hw->rcb_value = at_rcb_64;
    hw->dma_ord = at_dma_ord_enh;
    hw->dmar_block = at_dma_req_256;
    hw->dmaw_block = at_dma_req_256;
    
    hw->cmb_rrd = 4;
    hw->cmb_tpd = 4;
    hw->cmb_rx_timer = 1; //about 2us
    hw->cmb_tx_timer = 1; //about 2us
    hw->smb_timer = 100000 ; // about 200ms
    atomic_set(&adapter->irq_sem, 0);
    spin_lock_init(&adapter->stats_lock);
    spin_lock_init(&adapter->tx_lock);
    spin_lock_init(&adapter->mb_lock);

    return 0;
}

int
at_reset(struct at_adapter *adapter)
{
    int ret;
    
    if (AT_SUCCESS != (ret = at_reset_hw(&adapter->hw)))
        return ret;

    return at_init_hw(&adapter->hw);
}

/**
 * at_open - Called when a network interface is made active
 * @netdev: network interface device structure
 *
 * Returns 0 on success, negative value on failure
 *
 * The open entry point is called when a network interface is made
 * active by the system (IFF_UP).  At this point all resources needed
 * for transmit and receive operations are allocated, the interrupt
 * handler is registered with the OS, the watchdog timer is started,
 * and the stack is notified that the interface is ready.
 **/

static int
at_open(struct net_device *netdev)
{
    struct at_adapter *adapter = netdev_priv(netdev);
    int err;

    DEBUGFUNC("at_open !");

    /* allocate transmit descriptors */

    if((err = at_setup_ring_resources(adapter)))
        return err;

    if((err = at_up(adapter)))
        goto err_up;
/* for test only */
/*    { uint32_t value;
       value = AT_READ_REG(&adapter->hw, 0x12FC);
       DEBUGOUT1("reg 0x12fc = 0x%x", value);
       value = AT_READ_REG(&adapter->hw, 0x1008);
       DEBUGOUT1("reg 0x1008 = 0x%x", value);
    }
*/
/**********end ****/



    return 0;

err_up:
    at_reset(adapter);

    return err;
}

/**
 * at_close - Disables a network interface
 * @netdev: network interface device structure
 *
 * Returns 0, this is not allowed to fail
 *
 * The close entry point is called when an interface is de-activated
 * by the OS.  The hardware is still under the drivers control, but
 * needs to be disabled.  A global MAC reset is issued to stop the
 * hardware, and all transmit and receive resources are freed.
 **/

static int
at_close(struct net_device *netdev)
{
    struct at_adapter *adapter = netdev_priv(netdev);
    DEBUGFUNC("at_close!");

    at_down(adapter);

    at_free_ring_resources(adapter);

    return 0;
}

/**
 * at_setup_mem_resources - allocate Tx / RX descriptor resources 
 * @adapter: board private structure
 *
 * Return 0 on success, negative on failure
 **/

int32_t
at_setup_ring_resources(struct at_adapter *adapter)
{
    struct at_tpd_ring *tpd_ring = &adapter->tpd_ring;
    struct at_rfd_ring *rfd_ring = &adapter->rfd_ring;
    struct at_rrd_ring *rrd_ring = &adapter->rrd_ring;
    struct at_ring_header * ring_header = &adapter->ring_header;
    struct pci_dev *pdev = adapter->pdev;
    int size;
    uint8_t offset = 0;

//    DEBUGFUNC("at_setup_ring_resources !");

//    DEBUGOUT1("TPD count = %x  RFD cont = %x  RRD = %x", 
//        tpd_ring->count, rfd_ring->count, rrd_ring->count);
        
    DEBUGOUT1("sz TPD:%d, sz RFD:%d, sz RRD:%d", 
		    sizeof(tx_packet_desc_t),
		    sizeof(rx_free_desc_t),
		    sizeof(rx_return_desc_t));

    size = sizeof(struct at_buffer) * (tpd_ring->count+rfd_ring->count);
    tpd_ring->buffer_info = kmalloc(size, GFP_KERNEL);
    if(!tpd_ring->buffer_info) {
    DEBUGOUT1("kmalloc failed , size = D%d", size);
        return -ENOMEM;
    }
    rfd_ring->buffer_info = 
         (struct at_buffer*)(tpd_ring->buffer_info+tpd_ring->count);

    memset(tpd_ring->buffer_info, 0, size);

    /* real ring DMA buffer */
    ring_header->size = size =  sizeof(tx_packet_desc_t ) * tpd_ring->count 
                              + sizeof(rx_free_desc_t) *    rfd_ring->count 
                              + sizeof(rx_return_desc_t) *  rrd_ring->count
                              + sizeof(coals_msg_block_t)
                              + sizeof(stats_msg_block_t)
                              + 40; // 40: for 8 bytes align
         
    ring_header->desc = 
          pci_alloc_consistent(pdev, ring_header->size, &ring_header->dma);
    if(!ring_header->desc) {
        kfree(tpd_ring->buffer_info);
        DEBUGOUT1("pci_alloc_consistent failed , size = D%d", size);
        return -ENOMEM;
    }
    
    if (adapter->pci_using_64) 
    { // test whether HIDWORD dma buffer is not cross boundary
        if (    ((ring_header->dma       &0xffffffff00000000ULL)>>32)
             != (((ring_header->dma+size)&0xffffffff00000000ULL)>>32) )
        {
            kfree(tpd_ring->buffer_info);
            pci_free_consistent(
                     pdev, 
                     ring_header->size, 
                     ring_header->desc, 
                     ring_header->dma);
            DEBUGOUT("memory allocated cross 32bit boundary !");
            return -ENOMEM;
        }
    }

//    DEBUGOUT("memory allocated successfully !");    
    
    memset(ring_header->desc, 0, ring_header->size);
    DEBUGOUT1("whole ring dma addr=%x", ring_header->dma);

    // init TPD ring
    tpd_ring->dma = ring_header->dma;
    offset = (tpd_ring->dma & 0x7) ? (8 - (ring_header->dma & 0x7)) : 0;
    tpd_ring->dma += offset;
    tpd_ring->desc = (uint8_t*) ring_header->desc + offset;
    tpd_ring->size = sizeof(tx_packet_desc_t) * tpd_ring->count;
    atomic_set(&tpd_ring->next_to_use, 0);
    atomic_set(&tpd_ring->next_to_clean, 0);
    DEBUGOUT1("tpd ring dma addr=%x", tpd_ring->dma);
    
    // init RFD ring
    rfd_ring->dma = tpd_ring->dma + tpd_ring->size;
    offset = (rfd_ring->dma & 0x7) ? (8 - (rfd_ring->dma & 0x7)) : 0;
    rfd_ring->dma += offset;
    rfd_ring->desc = (uint8_t*) tpd_ring->desc + (tpd_ring->size+offset);
    rfd_ring->size = sizeof(rx_free_desc_t) * rfd_ring->count;
        
    rfd_ring->next_to_clean = 0;
    //rfd_ring->next_to_use = rfd_ring->count - 1;
    atomic_set(&rfd_ring->next_to_use, 0);
    DEBUGOUT1("rfd ring dma addr=%x", rfd_ring->dma);
    
    // init RRD ring
    rrd_ring->dma = rfd_ring->dma + rfd_ring->size;
    offset = (rrd_ring->dma & 0x7) ? (8 - (rrd_ring->dma & 0x7)) : 0;
    rrd_ring->dma += offset;
    rrd_ring->desc = (uint8_t*) rfd_ring->desc + (rfd_ring->size+offset);
    rrd_ring->size = sizeof(rx_return_desc_t) * rrd_ring->count;
    rrd_ring->next_to_use = 0;
    atomic_set(&rrd_ring->next_to_clean, 0);
    DEBUGOUT1("rrd ring dma addr=%x", rrd_ring->dma);
    
    // init CMB
    adapter->cmb.dma = rrd_ring->dma + rrd_ring->size;
    offset = (adapter->cmb.dma & 0x7)? (8-(adapter->cmb.dma & 0x7)) : 0;
    adapter->cmb.dma += offset;
    adapter->cmb.cmb = 
       (coals_msg_block_t*)
          ((uint8_t*)rrd_ring->desc + (rrd_ring->size+offset));
    DEBUGOUT1("cmd dma addr=%x", adapter->cmb.dma);
    
    // init SMB
    adapter->smb.dma = adapter->cmb.dma + sizeof(coals_msg_block_t);
    offset = (adapter->smb.dma&0x7) ? (8-(adapter->smb.dma&0x7)): 0;
    adapter->smb.dma += offset;
    adapter->smb.smb = 
	    (stats_msg_block_t*)
          ((uint8_t*)adapter->cmb.cmb + (sizeof(coals_msg_block_t)+offset));
    DEBUGOUT1("smb dma addr=%x", adapter->smb.dma);

    return AT_SUCCESS;
}

/**
 * at_free_ring_resources - Free Tx / RX descriptor Resources
 * @adapter: board private structure
 *
 * Free all transmit software resources
 **/

void
at_free_ring_resources(struct at_adapter *adapter)
{
    struct pci_dev *pdev = adapter->pdev;
    struct at_tpd_ring *tpd_ring = &adapter->tpd_ring; 
    struct at_rfd_ring *rfd_ring = &adapter->rfd_ring;
    struct at_rrd_ring * rrd_ring = &adapter->rrd_ring;
    struct at_ring_header * ring_header = &adapter->ring_header;
    
//    DEBUGFUNC("at_free_ring_resources !");

    at_clean_tx_ring(adapter);
    at_clean_rx_ring(adapter);
    
    kfree(tpd_ring->buffer_info);
    pci_free_consistent(
         pdev, 
         ring_header->size, 
         ring_header->desc, 
         ring_header->dma);
    
    tpd_ring->buffer_info = NULL;
    tpd_ring->desc = NULL;
    tpd_ring->dma = 0;
    
    rfd_ring->buffer_info = NULL;
    rfd_ring->desc = NULL;
    rfd_ring->dma = 0;
    
    rrd_ring->desc = NULL;
    rrd_ring->dma = 0;
}


int32_t
at_up(struct at_adapter *adapter)
{
    struct net_device *netdev = adapter->netdev;
    int err;

    DEBUGFUNC("at_up !"); 
    
    /* hardware has been reset, we need to reload some things */

    at_set_multi(netdev);

#ifdef NETIF_F_HW_VLAN_TX
    at_restore_vlan(adapter);
#endif

    err = at_alloc_rx_buffers(adapter);
    if (0 == err)
    { // no RX BUFFER allocated
        return -ENOMEM;
    }    
    
    if (at_configure(adapter)) {
        err = -EIO;
        goto err_up;
    }
   
    if ((err = request_irq(adapter->pdev->irq, 
                          &at_intr,
                          SA_SHIRQ | SA_SAMPLE_RANDOM,
                          netdev->name, netdev)))
        goto err_up;
        
    mod_timer(&adapter->watchdog_timer, jiffies); 
    
    at_irq_enable(adapter);
    
    at_check_link(adapter);

    return 0;

    // free irq
    // disable any interrupt
    AT_WRITE_REG(&adapter->hw, REG_IMR, 0);
    free_irq(adapter->pdev->irq, netdev);
    
err_up:
   
    // free rx_buffers
    at_clean_rx_ring(adapter);
    return err;    
}

inline  void
at_setup_mac_ctrl(struct at_adapter* adapter)
{
    uint32_t value;
    struct at_hw* hw = &adapter->hw;
    struct net_device* netdev = adapter->netdev;
    
    /* Config MAC CTRL Register */
    value = MAC_CTRL_TX_EN | MAC_CTRL_RX_EN;
    // duplex
    if (FULL_DUPLEX == adapter->link_duplex)    
        value |= MAC_CTRL_DUPLX;
    // speed
    value |= ((uint32_t)((SPEED_1000 == adapter->link_speed) ? 
              MAC_CTRL_SPEED_1000 : 
              MAC_CTRL_SPEED_10_100)<< MAC_CTRL_SPEED_SHIFT);
    // flow control
    value |= (MAC_CTRL_TX_FLOW|MAC_CTRL_RX_FLOW);

    // PAD & CRC
    value |= (MAC_CTRL_ADD_CRC|MAC_CTRL_PAD);
    // preamble length
    value |= (((uint32_t)adapter->hw.preamble_len
                  &MAC_CTRL_PRMLEN_MASK)<< MAC_CTRL_PRMLEN_SHIFT);
    // vlan 
    if (adapter->vlgrp)     
        value |= MAC_CTRL_RMV_VLAN;
/*        
    // rx checksum 
    if (adapter->rx_csum)
        value |= MAC_CTRL_RX_CHKSUM_EN;
*/        
        
    // filter mode
    value |= MAC_CTRL_BC_EN;
    if (netdev->flags & IFF_PROMISC) 
        value |= MAC_CTRL_PROMIS_EN;
    else if (netdev->flags & IFF_ALLMULTI)
        value |= MAC_CTRL_MC_ALL_EN;

//        value |= MAC_CTRL_LOOPBACK;
    AT_WRITE_REG(hw, REG_MAC_CTRL, value);
}
static uint32_t
at_check_link(struct at_adapter* adapter)
{
    struct at_hw *hw = &adapter->hw;
    struct net_device * netdev = adapter->netdev;
    uint32_t ret_val;
    uint16_t speed, duplex, phy_data;
    int reconfig = 0;

//    DEBUGFUNC("at_check_link !");
	// MII_BMSR must read twise
    at_read_phy_reg(hw, MII_BMSR, &phy_data);
    at_read_phy_reg(hw, MII_BMSR, &phy_data);
    if (!(phy_data&BMSR_LSTATUS)) { // link down
		if (netif_carrier_ok(netdev)) { // old link state: Up
			DEBUGOUT("NIC Link is Down");
            adapter->link_speed = SPEED_0;
            netif_carrier_off(netdev);
            netif_stop_queue(netdev);
        }
        return AT_SUCCESS;  
    }
    
    // Link Up
	ret_val = at_get_speed_and_duplex(hw, &speed, &duplex);
	if (ret_val)  return ret_val;
	switch( hw->MediaType )
	{
	case MEDIA_TYPE_1000M_FULL:
		if (speed  != SPEED_1000 || duplex != FULL_DUPLEX)	
			reconfig = 1;
		break;
	case MEDIA_TYPE_100M_FULL:
		if (speed  != SPEED_100 || duplex != FULL_DUPLEX)
			reconfig = 1;
		break;
	case MEDIA_TYPE_100M_HALF:
		if (speed  != SPEED_100 || duplex != HALF_DUPLEX)
			reconfig = 1;
		break;
	case MEDIA_TYPE_10M_FULL:
		if (speed != SPEED_10 || duplex != FULL_DUPLEX)
			reconfig = 1;
	        break;	
	case MEDIA_TYPE_10M_HALF:
		if (speed  != SPEED_10 || duplex != HALF_DUPLEX)
			reconfig = 1;
		break;
	}
	// link result is our setting
	if (0 == reconfig)
	{
		if (adapter->link_speed != speed ||
            adapter->link_duplex != duplex ) {
			adapter->link_speed = speed;
			adapter->link_duplex = duplex;
			at_setup_mac_ctrl(adapter); 
			printk(KERN_INFO
                   "%s: %s NIC Link is Up<%d Mbps %s>\n",
		   			at_driver_name,
                    netdev->name, adapter->link_speed,
                    adapter->link_duplex == FULL_DUPLEX ?
 					"Full Duplex" : "Half Duplex"); 
		}
		
		if (!netif_carrier_ok(netdev)) { // Link down -> Up
			netif_carrier_on(netdev);
			netif_wake_queue(netdev);
		}
		return AT_SUCCESS;
	}
	
	// change orignal link status
	if (netif_carrier_ok(netdev)) { 
		adapter->link_speed = SPEED_0;
    	netif_carrier_off(netdev);
    	netif_stop_queue(netdev);
    }
    
    if (hw->MediaType != MEDIA_TYPE_AUTO_SENSOR &&
    	hw->MediaType != MEDIA_TYPE_1000M_FULL ) {
    	switch (hw->MediaType)
    	{
    	case MEDIA_TYPE_100M_FULL:
    		phy_data = MII_CR_FULL_DUPLEX|MII_CR_SPEED_100|MII_CR_RESET;
    		break;
    	case MEDIA_TYPE_100M_HALF:
    		phy_data = MII_CR_SPEED_100|MII_CR_RESET;
    		break;
    	case MEDIA_TYPE_10M_FULL:
    		phy_data = MII_CR_FULL_DUPLEX|MII_CR_SPEED_10|MII_CR_RESET;
    		break;
    	default: // MEDIA_TYPE_10M_HALF:
    		phy_data = MII_CR_SPEED_10|MII_CR_RESET;
    		break;
    	}
    	at_write_phy_reg(hw, MII_BMCR, phy_data);
DEBUGOUT("RE-CONFIG-PHY !");
    	return AT_SUCCESS;
    }

	// auto-neg, insert timer to re-config phy
    if (!adapter->phy_timer_pending) {
		adapter->phy_timer_pending = TRUE;
		mod_timer(&adapter->phy_config_timer, jiffies + 3 * HZ);
	}

    return AT_SUCCESS;
}


void
at_down(struct at_adapter *adapter)
{
    struct net_device *netdev = adapter->netdev;
    
    DEBUGFUNC("at_down !");

    del_timer_sync(&adapter->watchdog_timer);
    del_timer_sync(&adapter->phy_config_timer);
    adapter->phy_timer_pending = FALSE;

    at_irq_disable(adapter);    
    free_irq(adapter->pdev->irq, netdev);
    at_reset_hw(&adapter->hw);
    adapter->cmb.cmb->int_stats = 0;

    adapter->link_speed = SPEED_0;
    adapter->link_duplex = -1;
    netif_carrier_off(netdev);
    netif_stop_queue(netdev);

    at_clean_tx_ring(adapter);
    at_clean_rx_ring(adapter);
}

/**
 * at_set_multi - Multicast and Promiscuous mode set
 * @netdev: network interface device structure
 *
 * The set_multi entry point is called whenever the multicast address
 * list or the network interface flags are updated.  This routine is
 * responsible for configuring the hardware for proper multicast,
 * promiscuous mode, and all-multi behavior.
 **/

static void
at_set_multi(struct net_device *netdev)
{
    struct at_adapter *adapter = netdev_priv(netdev);
    struct at_hw *hw = &adapter->hw;
    struct dev_mc_list *mc_ptr;
    uint32_t rctl;
    uint32_t hash_value;

//    DEBUGFUNC("at_set_multi !");

    /* Check for Promiscuous and All Multicast modes */

    rctl = AT_READ_REG(hw, REG_MAC_CTRL);

    if(netdev->flags & IFF_PROMISC) {
        rctl |= MAC_CTRL_PROMIS_EN;
    } else if(netdev->flags & IFF_ALLMULTI) {
        rctl |= MAC_CTRL_MC_ALL_EN;
        rctl &= ~MAC_CTRL_PROMIS_EN;
    } else {
        rctl &= ~(MAC_CTRL_PROMIS_EN | MAC_CTRL_MC_ALL_EN);
    }

    AT_WRITE_REG(hw, REG_MAC_CTRL, rctl);

    /* clear the old settings from the multicast hash table */
    AT_WRITE_REG(hw, REG_RX_HASH_TABLE, 0);
    AT_WRITE_REG_ARRAY(hw, REG_RX_HASH_TABLE, 1, 0);

    /* comoute mc addresses' hash value ,and put it into hash table */

    for(mc_ptr = netdev->mc_list; mc_ptr; mc_ptr = mc_ptr->next) {
        hash_value = at_hash_mc_addr(hw, mc_ptr->dmi_addr);
        at_hash_set(hw, hash_value);
    }
}

#ifdef NETIF_F_HW_VLAN_TX
static void
at_vlan_rx_register(struct net_device *netdev, struct vlan_group *grp)
{
    struct at_adapter *adapter = netdev_priv(netdev);
    uint32_t ctrl;

 //   DEBUGFUNC("at_vlan_rx_register !");    

    at_irq_disable(adapter);
    adapter->vlgrp = grp;

    if(grp) {
        /* enable VLAN tag insert/strip */

        ctrl = AT_READ_REG(&adapter->hw, REG_MAC_CTRL);
        ctrl |= MAC_CTRL_RMV_VLAN; 
        AT_WRITE_REG(&adapter->hw, REG_MAC_CTRL, ctrl);
    } else {
        /* disable VLAN tag insert/strip */

        ctrl = AT_READ_REG(&adapter->hw, REG_MAC_CTRL);
        ctrl &= ~MAC_CTRL_RMV_VLAN;
        AT_WRITE_REG(&adapter->hw, REG_MAC_CTRL, ctrl);
    }

    at_irq_enable(adapter);
}

static void
at_vlan_rx_add_vid(struct net_device *netdev, uint16_t vid)
{
    /* We don't do Vlan filtering */
//    DEBUGFUNC("at_vlan_rx_add_vid !");
    return ;
}

static void
at_vlan_rx_kill_vid(struct net_device *netdev, uint16_t vid)
{
    struct at_adapter *adapter = netdev_priv(netdev);

//    DEBUGFUNC("at_vlan_rx_kill_vid !");
    at_irq_disable(adapter);

    if(adapter->vlgrp)
        adapter->vlgrp->vlan_devices[vid] = NULL;

    at_irq_enable(adapter);

    /* We don't do Vlan filtering */

    return;
}

static void
at_restore_vlan(struct at_adapter *adapter)
{
//    DEBUGFUNC("at_restore_vlan !");
    at_vlan_rx_register(adapter->netdev, adapter->vlgrp);

    if(adapter->vlgrp) {
        uint16_t vid;
        for(vid = 0; vid < VLAN_GROUP_ARRAY_LEN; vid++) {
            if(!adapter->vlgrp->vlan_devices[vid])
                continue;
            at_vlan_rx_add_vid(adapter->netdev, vid);
        }
    }
}
#endif

static inline
void set_flow_ctrl_old(struct at_adapter* adapter)
{
    uint32_t Hi, Lo, value;
	
    // RFD Flow Control   
    value = adapter->rfd_ring.count;
    Hi = value / 16;
    if (Hi < 2) Hi = 2;
    Lo = value * 7 / 8;
    
   value = ((Hi&RXQ_RXF_PAUSE_TH_HI_MASK) << RXQ_RXF_PAUSE_TH_HI_SHIFT)|
	((Lo&RXQ_RXF_PAUSE_TH_LO_MASK) << RXQ_RXF_PAUSE_TH_LO_SHIFT);
    AT_WRITE_REG(&adapter->hw, REG_RXQ_RXF_PAUSE_THRESH, value);
    
    // RRD Flow Control
    value = adapter->rrd_ring.count;
    Lo = value / 16;
    Hi = value * 7 / 8;
    if (Lo < 2) Lo = 2;

   value = ((Hi&RXQ_RRD_PAUSE_TH_HI_MASK) << RXQ_RRD_PAUSE_TH_HI_SHIFT)|
	((Lo&RXQ_RRD_PAUSE_TH_LO_MASK) << RXQ_RRD_PAUSE_TH_LO_SHIFT);
    AT_WRITE_REG(&adapter->hw, REG_RXQ_RRD_PAUSE_THRESH, value);
}

static inline
void set_flow_ctrl_new(struct at_hw* hw)
{
    uint32_t Hi, Lo, value;
	
    // RXF Flow Control   
    value = AT_READ_REG(hw, REG_SRAM_RXF_LEN);
    Lo = value / 16;
    if (Lo < 192) Lo = 192;
    Hi = value * 7 / 8;
    if (Hi < Lo) Hi = Lo + 16;
   value = ((Hi&RXQ_RXF_PAUSE_TH_HI_MASK) << RXQ_RXF_PAUSE_TH_HI_SHIFT)|
	((Lo&RXQ_RXF_PAUSE_TH_LO_MASK) << RXQ_RXF_PAUSE_TH_LO_SHIFT);
    AT_WRITE_REG(hw, REG_RXQ_RXF_PAUSE_THRESH, value);
    
    // RRD Flow Control
    value = AT_READ_REG(hw, REG_SRAM_RRD_LEN);
    Lo = value / 8;
    Hi = value * 7 / 8;
    if (Lo < 2) Lo = 2;
    if (Hi < Lo) Hi = Lo + 3;
   value = ((Hi&RXQ_RRD_PAUSE_TH_HI_MASK) << RXQ_RRD_PAUSE_TH_HI_SHIFT)|
	((Lo&RXQ_RRD_PAUSE_TH_LO_MASK) << RXQ_RRD_PAUSE_TH_LO_SHIFT);
    AT_WRITE_REG(hw, REG_RXQ_RRD_PAUSE_THRESH, value);
}

/**
 * at_configure - Configure Transmit&Receive Unit after Reset
 * @adapter: board private structure
 *
 * Configure the Tx /Rx unit of the MAC after a reset.
 **/

static uint32_t
at_configure(struct at_adapter *adapter)
{
    struct at_hw * hw = &adapter->hw;
    uint32_t value;
    
//  DEBUGFUNC("at_configure !");
    // clear interrupt status
    AT_WRITE_REG(&adapter->hw, REG_ISR, 0xffffffff);

    // set MAC Address
    value = (((uint32_t)hw->mac_addr[2]) << 24) |
            (((uint32_t)hw->mac_addr[3]) << 16) |
            (((uint32_t)hw->mac_addr[4]) << 8 ) |
            (((uint32_t)hw->mac_addr[5])      ) ;
    AT_WRITE_REG(hw, REG_MAC_STA_ADDR, value);
    value = (((uint32_t)hw->mac_addr[0]) << 8 ) |
            (((uint32_t)hw->mac_addr[1])      ) ;
    AT_WRITE_REG(hw, (REG_MAC_STA_ADDR+4), value);

    // tx / rx ring :
    
    // HI base address
    AT_WRITE_REG(
          hw, 
          REG_DESC_BASE_ADDR_HI, 
          (uint32_t)((adapter->tpd_ring.dma&0xffffffff00000000ULL) >>32));
    // LO base address
    AT_WRITE_REG(
          hw, 
          REG_DESC_RFD_ADDR_LO, 
          (uint32_t)(adapter->rfd_ring.dma& 0x00000000ffffffffULL));
    AT_WRITE_REG(
          hw, 
          REG_DESC_RRD_ADDR_LO, 
          (uint32_t)(adapter->rrd_ring.dma& 0x00000000ffffffffULL));
    AT_WRITE_REG(hw, 
                 REG_DESC_TPD_ADDR_LO, 
                 (uint32_t)(adapter->tpd_ring.dma& 0x00000000ffffffffULL));
    AT_WRITE_REG(hw, 
                 REG_DESC_CMB_ADDR_LO, 
                 (uint32_t)(adapter->cmb.dma& 0x00000000ffffffffULL));
    AT_WRITE_REG(hw, 
                 REG_DESC_SMB_ADDR_LO, 
                 (uint32_t)(adapter->smb.dma& 0x00000000ffffffffULL));
    
    // element count
    value = adapter->rrd_ring.count;
    value <<= 16;
    value += adapter->rfd_ring.count;
    AT_WRITE_REG(hw, REG_DESC_RFD_RRD_RING_SIZE, value);
    AT_WRITE_REG(hw, REG_DESC_TPD_RING_SIZE, adapter->tpd_ring.count);

/* 
    // config SRAM
    // add RXF 256*8 bytes   
    value = ((2795 + 256) << 16) | 432;
    AT_WRITE_REG(hw, REG_SRAM_RXF_ADDR, value);
    value = 2364 + 256;
    AT_WRITE_REG(hw, REG_SRAM_RXF_LEN, value);
    // sub TXF 256*8 bytes
    value = (4075 << 16) | (2796 + 256);
    AT_WRITE_REG(hw, REG_SRAM_TXF_ADDR, value);
    value = 1280 - 256;
    AT_WRITE_REG(hw, REG_SRAM_TXF_LEN, value);
*/
    // Load Ptr
    AT_WRITE_REG(hw, REG_LOAD_PTR, 1);
    
    
    /* config Mailbox */
    
    value = 
         (((uint32_t)atomic_read(&adapter->tpd_ring.next_to_use)
            &MB_TPD_PROD_INDX_MASK)<<MB_TPD_PROD_INDX_SHIFT) |
         (((uint32_t)atomic_read(&adapter->rrd_ring.next_to_clean)
            &MB_RRD_CONS_INDX_MASK)<<MB_RRD_CONS_INDX_SHIFT) |
         (((uint32_t)atomic_read(&adapter->rfd_ring.next_to_use)
            &MB_RFD_PROD_INDX_MASK)<<MB_RFD_PROD_INDX_SHIFT);
    AT_WRITE_REG(hw, REG_MAILBOX, value);
    
//    DEBUGOUT1("init Mailbox with 0x%x", value);
    
    /* config IPG/IFG */
    value = 
        (((uint32_t)hw->ipgt&MAC_IPG_IFG_IPGT_MASK) 
              <<MAC_IPG_IFG_IPGT_SHIFT) |
        (((uint32_t)hw->min_ifg &MAC_IPG_IFG_MIFG_MASK) 
              <<MAC_IPG_IFG_MIFG_SHIFT) |
        (((uint32_t)hw->ipgr1&MAC_IPG_IFG_IPGR1_MASK)
              <<MAC_IPG_IFG_IPGR1_SHIFT)|
        (((uint32_t)hw->ipgr2&MAC_IPG_IFG_IPGR2_MASK)
              <<MAC_IPG_IFG_IPGR2_SHIFT);
    AT_WRITE_REG(hw, REG_MAC_IPG_IFG, value);
//    DEBUGOUT1("init ipg/ifg with 0x%x", value);
    
    /* config  Half-Duplex Control */
    value = 
      ((uint32_t)hw->lcol&MAC_HALF_DUPLX_CTRL_LCOL_MASK) |
      (((uint32_t)hw->max_retry&MAC_HALF_DUPLX_CTRL_RETRY_MASK)
          <<MAC_HALF_DUPLX_CTRL_RETRY_SHIFT) |
      MAC_HALF_DUPLX_CTRL_EXC_DEF_EN   |
      (0xa<<MAC_HALF_DUPLX_CTRL_ABEBT_SHIFT) |
      (((uint32_t)hw->jam_ipg&MAC_HALF_DUPLX_CTRL_JAMIPG_MASK)
          <<MAC_HALF_DUPLX_CTRL_JAMIPG_SHIFT);
    AT_WRITE_REG(hw, REG_MAC_HALF_DUPLX_CTRL, value);
//    DEBUGOUT1("init Half Duplex with 0x%x", value);
    
    
    /* set Interrupt Moderator Timer */
    AT_WRITE_REGW(hw, REG_IRQ_MODU_TIMER_INIT, adapter->imt);
    AT_WRITE_REG(hw, REG_MASTER_CTRL, MASTER_CTRL_ITIMER_EN);
//    DEBUGOUT1("init Irq Modurator Timer with 0x%x", adapter->imt);
    
    /* set Interrupt Clear Timer */
    AT_WRITE_REGW(hw, REG_CMBDISDMA_TIMER, adapter->ict);
//    DEBUGOUT1("init Irq Clear Timer with 0x%x", adapter->ict);
    
    /* set MTU */
    AT_WRITE_REG(hw, REG_MTU, hw->max_frame_size+4); // 4 : VLAN
//    DEBUGOUT1("init MTU with 0x%x", hw->max_frame_size); 

    // jumbo size & rrd retirement timer
    value = 
      (((uint32_t)hw->rx_jumbo_th&RXQ_JMBOSZ_TH_MASK) 
          << RXQ_JMBOSZ_TH_SHIFT)|
      (((uint32_t)hw->rx_jumbo_lkah&RXQ_JMBO_LKAH_MASK) 
          << RXQ_JMBO_LKAH_SHIFT)|
      (((uint32_t)hw->rrd_ret_timer&RXQ_RRD_TIMER_MASK)
          << RXQ_RRD_TIMER_SHIFT) ;
    AT_WRITE_REG(hw, REG_RXQ_JMBOSZ_RRDTIM, value);
//    DEBUGOUT1("init RXQ Jumbo size RRD retirement Timer with 0x=%x", value);
    // Flow Control
    switch (hw->dev_rev)	    
    {
    case 0x8001: 
    case 0x9001:
    case 0x9002:
    case 0x9003:
	    set_flow_ctrl_old(adapter);
	    break;
    default:
	    set_flow_ctrl_new(hw);
	    break;
    }
				
    /* config TXQ */
    value =  
       (((uint32_t)hw->tpd_burst&TXQ_CTRL_TPD_BURST_NUM_MASK)
           << TXQ_CTRL_TPD_BURST_NUM_SHIFT) |
       (((uint32_t)hw->txf_burst&TXQ_CTRL_TXF_BURST_NUM_MASK) 
           << TXQ_CTRL_TXF_BURST_NUM_SHIFT) |
       (((uint32_t)hw->tpd_fetch_th&TXQ_CTRL_TPD_FETCH_TH_MASK)
           << TXQ_CTRL_TPD_FETCH_TH_SHIFT) |
       TXQ_CTRL_ENH_MODE |
       TXQ_CTRL_EN;
    AT_WRITE_REG(hw, REG_TXQ_CTRL, value);
//    DEBUGOUT1("init TXQ Control with 0x%x", value); 
     
    // min tpd fetch gap & tx jumbo packet size threshold for taskoffload
    value = 
      (((uint32_t)hw->tx_jumbo_task_th&TX_JUMBO_TASK_TH_MASK)
         << TX_JUMBO_TASK_TH_SHIFT) |
      (((uint32_t)hw->tpd_fetch_gap&TX_TPD_MIN_IPG_MASK)
         << TX_TPD_MIN_IPG_SHIFT);
    AT_WRITE_REG(hw, REG_TX_JUMBO_TASK_TH_TPD_IPG, value);
//    DEBUGOUT1("init TPD fetch gap & TX jumbo taskoffload threshold with 0x%x", value);
    
    /* config RXQ */
    value =  
      (((uint32_t)hw->rfd_burst&RXQ_CTRL_RFD_BURST_NUM_MASK) 
          << RXQ_CTRL_RFD_BURST_NUM_SHIFT)|
      (((uint32_t)hw->rrd_burst&RXQ_CTRL_RRD_BURST_THRESH_MASK)
          << RXQ_CTRL_RRD_BURST_THRESH_SHIFT)|
      (((uint32_t)hw->rfd_fetch_gap&RXQ_CTRL_RFD_PREF_MIN_IPG_MASK)
          << RXQ_CTRL_RFD_PREF_MIN_IPG_SHIFT) |
      RXQ_CTRL_CUT_THRU_EN | 
      RXQ_CTRL_EN ;
      AT_WRITE_REG(hw, REG_RXQ_CTRL, value);
//     DEBUGOUT1("init RXQ control with 0x%x", value);
   
    /* config  DMA Engine */
    value = 
      ((((uint32_t)hw->dmar_block)&DMA_CTRL_DMAR_BURST_LEN_MASK) 
           << DMA_CTRL_DMAR_BURST_LEN_SHIFT)|
      ((((uint32_t)hw->dmaw_block)&DMA_CTRL_DMAR_BURST_LEN_MASK) 
           << DMA_CTRL_DMAR_BURST_LEN_SHIFT) |
      DMA_CTRL_DMAR_EN | 
      DMA_CTRL_DMAW_EN;
    value  |= (uint32_t)hw->dma_ord;
    if (at_rcb_128 == hw->rcb_value) 
    {
        value |= DMA_CTRL_RCB_VALUE;
    }
    AT_WRITE_REG(hw, REG_DMA_CTRL, value);
//    DEBUGOUT1("init DMA Engine with 0x%x", value);
    
    /* config CMB / SMB */
    value = hw->cmb_rrd | ((uint32_t)hw->cmb_tpd << 16);
    AT_WRITE_REG(hw, REG_CMB_WRITE_TH, value);
    value = hw->cmb_rx_timer | ((uint32_t)hw->cmb_tx_timer << 16);
    AT_WRITE_REG(hw, REG_CMB_WRITE_TIMER, value);
    AT_WRITE_REG(hw, REG_SMB_TIMER, hw->smb_timer);
//    DEBUGOUT1("init CMB Write Timer with 0x%x", value);

    // --- enable CMB / SMB 
    value = CSMB_CTRL_CMB_EN | CSMB_CTRL_SMB_EN;
    AT_WRITE_REG(hw, REG_CSMB_CTRL, value);
    
    value = AT_READ_REG(&adapter->hw, REG_ISR);
    if ((value&ISR_PHY_LINKDOWN) != 0) {
        value = 1; // config failed 
    } else {
        value = 0;
    }
    // clear all interrupt status
    AT_WRITE_REG(&adapter->hw, REG_ISR, 0x3fffffff);
    AT_WRITE_REG(&adapter->hw, REG_ISR, 0);
    return value;
}

/**
 * at_set_mac - Change the Ethernet Address of the NIC
 * @netdev: network interface device structure
 * @p: pointer to an address structure
 *
 * Returns 0 on success, negative on failure
 **/

static int
at_set_mac(struct net_device *netdev, void *p)
{
    struct at_adapter *adapter = netdev_priv(netdev);
    struct sockaddr *addr = p;

    DEBUGFUNC("at_set_mac !");
    
    if (netif_running(netdev))
        return -EBUSY; 

    if(!is_valid_ether_addr(addr->sa_data))
        return -EADDRNOTAVAIL;

    memcpy(netdev->dev_addr, addr->sa_data, netdev->addr_len);
    memcpy(adapter->hw.mac_addr, addr->sa_data, netdev->addr_len);
    
    set_mac_addr(&adapter->hw);


    return 0;
}



/**
 * at_change_mtu - Change the Maximum Transfer Unit
 * @netdev: network interface device structure
 * @new_mtu: new value for maximum frame size
 *
 * Returns 0 on success, negative on failure
 **/

static int
at_change_mtu(struct net_device *netdev, int new_mtu)
{
    struct at_adapter *adapter = netdev_priv(netdev);
    int old_mtu = netdev->mtu;
    int max_frame = new_mtu + ENET_HEADER_SIZE + ETHERNET_FCS_SIZE;

    DEBUGFUNC("at_change_mtu !");

    if((max_frame < MINIMUM_ETHERNET_FRAME_SIZE) ||
       (max_frame > MAX_JUMBO_FRAME_SIZE)) {
        AT_ERR("Invalid MTU setting\n");
        return -EINVAL;
    }


    adapter->hw.max_frame_size = max_frame;
    adapter->hw.tx_jumbo_task_th = (max_frame+7)>>3;
    adapter->rx_buffer_len = (max_frame+7) & ~7;
    
    adapter->hw.rx_jumbo_th = adapter->rx_buffer_len  / 8;
    
    netdev->mtu = new_mtu;
        
    if((old_mtu != new_mtu) && netif_running(netdev)) {
        
        at_down(adapter);
        at_up(adapter);
    }

    return 0;
}


/**
 * at_alloc_rx_buffers - Replace used receive buffers
 * @adapter: address of board private structure
 **/

static uint16_t
at_alloc_rx_buffers(struct at_adapter *adapter)
{
    struct at_rfd_ring *rfd_ring = &adapter->rfd_ring;
    struct net_device *netdev = adapter->netdev;
    struct pci_dev *pdev = adapter->pdev;
    struct page *page;                                                       
    unsigned long offset;                                                    
    struct at_buffer *buffer_info, * next_info;
    struct sk_buff *skb;
    uint16_t num_alloc = 0;
    uint16_t rfd_next_to_use, next_next;
    rx_free_desc_t *rfd_desc;

    next_next = rfd_next_to_use = (uint16_t)atomic_read(&rfd_ring->next_to_use);
    if (++next_next == rfd_ring->count)      next_next = 0;
    buffer_info = &rfd_ring->buffer_info[rfd_next_to_use];
    next_info = &rfd_ring->buffer_info[next_next];

    while (!buffer_info->alloced && !next_info->alloced) {
        if (NULL != buffer_info->skb) {
            buffer_info->alloced = 1;
	    DEBUGOUT1("skip rfd allocate (%d)", rfd_next_to_use);
            goto next;
        }
	
        rfd_desc = AT_RFD_DESC(rfd_ring, rfd_next_to_use);

        skb = dev_alloc_skb(adapter->rx_buffer_len + NET_IP_ALIGN);

        if(!skb) {
            /* Better luck next round */
            adapter->net_stats.rx_dropped++;
            break;
        }

        /* Make buffer alignment 2 beyond a 16 byte boundary
         * this will result in a 16 byte aligned IP header after
         * the 14 byte MAC header is removed
         */
        skb_reserve(skb, NET_IP_ALIGN);

        skb->dev = netdev;

        buffer_info->alloced = 1;
        buffer_info->skb = skb;
        buffer_info->length = (uint16_t)adapter->rx_buffer_len;
        page = virt_to_page(skb->data);                                            
        offset = (unsigned long) skb->data & ~PAGE_MASK;  
        buffer_info->dma =
                pci_map_page(pdev,
                             page,
                             offset,
                             adapter->rx_buffer_len,
                             PCI_DMA_FROMDEVICE);
        rfd_desc->buffer_addr = cpu_to_le64(buffer_info->dma);
        rfd_desc->buf_len = cpu_to_le16(adapter->rx_buffer_len);
        rfd_desc->coalese = 0;
        
next:
        rfd_next_to_use = next_next;
        if (++next_next == rfd_ring->count)     next_next = 0;
	
        buffer_info = &rfd_ring->buffer_info[rfd_next_to_use];
        next_info = &rfd_ring->buffer_info[next_next];
        num_alloc++;
    }
    
    if (0 != num_alloc) {
        /* Force memory writes to complete before letting h/w
         * know there are new descriptors to fetch.  (Only
         * applicable for weak-ordered memory model archs,
         * such as IA-64). */
        wmb();
        atomic_set(&rfd_ring->next_to_use, (int)rfd_next_to_use);
    }
    return num_alloc;
}


void
at_read_pci_cfg(struct at_hw *hw, uint32_t reg, uint16_t *value)
{
    struct at_adapter *adapter = hw->back;

    pci_read_config_word(adapter->pdev, reg, value);
}

void
at_write_pci_cfg(struct at_hw *hw, uint32_t reg, uint16_t *value)
{
    struct at_adapter *adapter = hw->back;

    pci_write_config_word(adapter->pdev, reg, *value);
}

/**
 * at_clean_tx_ring - Free Tx Buffers
 * @adapter: board private structure
 **/

static void
at_clean_tx_ring(struct at_adapter *adapter)
{
    struct at_tpd_ring *tpd_ring = &adapter->tpd_ring;
    struct at_buffer *buffer_info;
    struct pci_dev *pdev = adapter->pdev;
    unsigned long size;
    unsigned int i;

//    DEBUGFUNC("at_clean_tx_ring !");

    /* Free all the Tx ring sk_buffs */

    for(i = 0; i < tpd_ring->count; i++) {
        buffer_info = &tpd_ring->buffer_info[i];
        if (buffer_info->dma) {
            pci_unmap_page(pdev,
                           buffer_info->dma,
                           buffer_info->length,
                           PCI_DMA_TODEVICE);
            buffer_info->dma = 0;
        }
    }
    
    for(i = 0; i < tpd_ring->count; i++) {
        buffer_info = &tpd_ring->buffer_info[i];
        if(buffer_info->skb) {
            dev_kfree_skb_any(buffer_info->skb);
            buffer_info->skb = NULL;
        }
    }

    size = sizeof(struct at_buffer) * tpd_ring->count;
    memset(tpd_ring->buffer_info, 0, size);

    /* Zero out the descriptor ring */

    memset(tpd_ring->desc, 0, tpd_ring->size);

    atomic_set(&tpd_ring->next_to_use, 0);
    atomic_set(&tpd_ring->next_to_clean, 0);
}

/**
 * at_clean_rx_ring - Free RFD Buffers
 * @adapter: board private structure
 **/

static void
at_clean_rx_ring(struct at_adapter *adapter)
{
    struct at_rfd_ring *rfd_ring = &adapter->rfd_ring;
    struct at_rrd_ring *rrd_ring = &adapter->rrd_ring;
    struct at_buffer *buffer_info;
    struct pci_dev *pdev = adapter->pdev;
    unsigned long size;
    unsigned int i;

//    DEBUGFUNC("at_clean_rx_ring !");

    /* Free all the Rx ring sk_buffs */

    for(i = 0; i < rfd_ring->count; i++) {
        buffer_info = &rfd_ring->buffer_info[i];
        if(buffer_info->dma) {

            pci_unmap_page(pdev,
                           buffer_info->dma,
                           buffer_info->length,
                           PCI_DMA_FROMDEVICE);
            buffer_info->dma = 0;
        }
        if(buffer_info->skb) {
            dev_kfree_skb(buffer_info->skb);
            buffer_info->skb = NULL;
        }
    }

    size = sizeof(struct at_buffer) * rfd_ring->count;
    memset(rfd_ring->buffer_info, 0, size);

    /* Zero out the descriptor ring */

    memset(rfd_ring->desc, 0, rfd_ring->size);

    rfd_ring->next_to_clean = 0;
    atomic_set(&rfd_ring->next_to_use, 0);
    
    rrd_ring->next_to_use = 0;
    atomic_set(&rrd_ring->next_to_clean, 0);
}

/**
 * at_get_stats - Get System Network Statistics
 * @netdev: network interface device structure
 *
 * Returns the address of the device statistics structure.
 * The statistics are actually updated from the timer callback.
 **/

static struct net_device_stats *
at_get_stats(struct net_device *netdev)
{
    struct at_adapter *adapter = netdev_priv(netdev);
  
   // DEBUGFUNC("at_get_stats");
    return &adapter->net_stats;
}       

/**
 * at_ioctl -
 * @netdev:
 * @ifreq:
 * @cmd:
 **/

static int
at_ioctl(struct net_device *netdev, struct ifreq *ifr, int cmd)
{
//    DEBUGFUNC("at_ioctl !");
    switch (cmd) {
#ifdef SIOCGMIIPHY
    case SIOCGMIIPHY:
    case SIOCGMIIREG:
    case SIOCSMIIREG:
        return at_mii_ioctl(netdev, ifr, cmd);
#endif

/*
#ifdef ETHTOOL_OPS_COMPAT
	case SIOCETHTOOL:
		return at_ethtool_ioctl(ifr);
#endif
*/
	
	//michael add 2005-11-1
	case SIOCETHTOOL:
		return at_ethtool_ioctl(netdev,ifr);
#ifdef SIOCDEVPRIVATE
    case SIOCDEVPRIVATE:
        return at_priv_ioctl(netdev, ifr);
#endif

	default:
		return -EOPNOTSUPP;
	}
}

#ifdef SIOCGMIIPHY
/**
 * at_mii_ioctl -
 * @netdev:
 * @ifreq:
 * @cmd:
 **/

static int
at_mii_ioctl(struct net_device *netdev, struct ifreq *ifr, int cmd)
{
    struct at_adapter *adapter = netdev_priv(netdev);
    struct mii_ioctl_data *data = (struct mii_ioctl_data *)&ifr->ifr_data;
    unsigned long flags;

//    DEBUGFUNC("at_mii_ioctl !");

    switch (cmd) {
    case SIOCGMIIPHY:
        data->phy_id = 0;
        break;
    case SIOCGMIIREG:
        if (!capable(CAP_NET_ADMIN))
            return -EPERM;
        spin_lock_irqsave(&adapter->stats_lock, flags);
        if (at_read_phy_reg(&adapter->hw, data->reg_num & 0x1F, &data->val_out)) {
            spin_unlock_irqrestore(&adapter->stats_lock, flags);
            return -EIO;
        }
        spin_unlock_irqrestore(&adapter->stats_lock, flags);
        break;
    case SIOCSMIIREG:
        if (!capable(CAP_NET_ADMIN))
            return -EPERM;
        if (data->reg_num & ~(0x1F))
            return -EFAULT;
         
        spin_lock_irqsave(&adapter->stats_lock, flags);
	DEBUGOUT1("<at_mii_ioctl> write %x %x", 
			data->reg_num, 
			data->val_in);
        if (at_write_phy_reg(&adapter->hw, data->reg_num, data->val_in)) {
            spin_unlock_irqrestore(&adapter->stats_lock, flags);
            return -EIO;
        }
        // ......
        spin_unlock_irqrestore(&adapter->stats_lock, flags);
        break;
        
    default:
        return -EOPNOTSUPP;
    }
    return AT_SUCCESS;
}

#endif

/**
 * at_tx_timeout - Respond to a Tx Hang
 * @netdev: network interface device structure
 **/

static void
at_tx_timeout(struct net_device *netdev)
{
    struct at_adapter *adapter = netdev_priv(netdev);

    DEBUGFUNC("at_tx_timeout !");

    /* Do the reset outside of interrupt context */
    schedule_work(&adapter->tx_timeout_task);
}

/*
 * If TPD Buffer size equal to 0, PCIE DMAR_TO_INT
 * will assert. We do soft reset <0x1400=1> according 
 * with the SPEC. BUT, it seemes that PCIE or DMA 
 * state-machine will not be reset. DMAR_TO_INT will
 * assert again and again.
 */
static void
at_tx_timeout_task(struct net_device *netdev)
{
    struct at_adapter *adapter = netdev_priv(netdev);
    //struct at_hw* hw = &adapter->hw;
    
    //uint32_t val1, val2;
    
    DEBUGFUNC("at_tx_timeout_task !");

    
    netif_device_detach(netdev);
    
    /*******   disp debug info **********
    val1 = AT_READ_REG(hw, 0x15f0);
    DEBUGOUT1("<15f0> = 0x%x", val1);
    val1 = AT_READ_REG(hw, 0x1540);
    val2 = AT_READ_REG(hw, 0x1544);
    DEBUGOUT1("<1540> = 0x%x <1544> = 0x%x", val1, val2);
    val1 = AT_READ_REG(hw, 0x1548);
    val2 = AT_READ_REG(hw, 0x154c);
    DEBUGOUT1("<1548> = 0x%x <154c> = 0x%x", val1, val2);
    val1 = AT_READ_REG(hw, 0x1550);
    val2 = AT_READ_REG(hw, 0x1554);
    DEBUGOUT1("<1550> = 0x%x <1554> = 0x%x", val1, val2);
    val1 = AT_READ_REG(hw, 0x1558);
    val2 = AT_READ_REG(hw, 0x155c);
    DEBUGOUT1("<1558> = 0x%x <155c> = 0x%x", val1, val2);
    
    DEBUGOUT1("tpd next to clean %d, tpd next to use %d",
		(uint16_t) atomic_read(&adapter->tpd_ring.next_to_clean),
                (uint16_t) atomic_read(&adapter->tpd_ring.next_to_use));

    DEBUGOUT1("rfd ring: 0x%08x 0x%08x 0x%08x 0x%08x 0x%08x 0x%08x",
               *((uint32_t*)(adapter->rfd_ring.desc) + 0),
               *((uint32_t*)(adapter->rfd_ring.desc) + 1),
               *((uint32_t*)(adapter->rfd_ring.desc) + 2),
               *((uint32_t*)(adapter->rfd_ring.desc) + 3),
               *((uint32_t*)(adapter->rfd_ring.desc) + 4),
               *((uint32_t*)(adapter->rfd_ring.desc) + 5));

    DEBUGOUT1("rfd ring: 0x%08x 0x%08x 0x%08x 0x%08x 0x%08x 0x%08x",
               *((uint32_t*)(adapter->rfd_ring.desc) + 6),
               *((uint32_t*)(adapter->rfd_ring.desc) + 7),
               *((uint32_t*)(adapter->rfd_ring.desc) + 8),
               *((uint32_t*)(adapter->rfd_ring.desc) + 9),
               *((uint32_t*)(adapter->rfd_ring.desc) + 10),
               *((uint32_t*)(adapter->rfd_ring.desc) + 11));
    
    DEBUGOUT1("rfd ring: 0x%08x 0x%08x 0x%08x 0x%08x 0x%08x 0x%08x",
               *((uint32_t*)(adapter->rfd_ring.desc) + 12),
               *((uint32_t*)(adapter->rfd_ring.desc) + 13),
               *((uint32_t*)(adapter->rfd_ring.desc) + 14),
               *((uint32_t*)(adapter->rfd_ring.desc) + 15),
               *((uint32_t*)(adapter->rfd_ring.desc) + 16),
               *((uint32_t*)(adapter->rfd_ring.desc) + 17));
    
    DEBUGOUT1("rfd ring: 0x%08x 0x%08x 0x%08x 0x%08x 0x%08x 0x%08x",
               *((uint32_t*)(adapter->rfd_ring.desc) + 18),
               *((uint32_t*)(adapter->rfd_ring.desc) + 19),
               *((uint32_t*)(adapter->rfd_ring.desc) + 20),
               *((uint32_t*)(adapter->rfd_ring.desc) + 21),
               *((uint32_t*)(adapter->rfd_ring.desc) + 22),
               *((uint32_t*)(adapter->rfd_ring.desc) + 23));
    */
    at_down(adapter);
    at_up(adapter);
    netif_device_attach(netdev);
}
/**
 * at_link_chg_task - deal with link change event Out of interrupt context
 * @netdev: network interface device structure
 **/
static void
at_link_chg_task(struct net_device* netdev)
{
    struct at_adapter * adapter = netdev_priv(netdev);
    unsigned long flags;
    DEBUGFUNC("at_link_chg_task !");
    
    spin_lock_irqsave(&adapter->stats_lock, flags);
    
    at_check_link(adapter);
    
    spin_unlock_irqrestore(&adapter->stats_lock, flags);
}

static void
at_check_for_link(struct at_adapter* adapter)
{
    struct net_device *netdev = adapter->netdev;
    uint16_t phy_data = 0;

    DEBUGFUNC("at_check_for_link!");
    
    spin_lock(&adapter->stats_lock);
    adapter->phy_timer_pending = FALSE;
    at_read_phy_reg(&adapter->hw, MII_BMSR, &phy_data);
    at_read_phy_reg(&adapter->hw, MII_BMSR, &phy_data);
    spin_unlock(&adapter->stats_lock);
    
    DEBUGOUT1("MII_BMSR=%x <at_check_for_link>", phy_data);
    
    // notify upper layer link down ASAP
    if (!(phy_data&BMSR_LSTATUS)) { // Link Down
        if (netif_carrier_ok(netdev)) { // old link state: Up
            printk(KERN_INFO
                   "%s: %s NIC Link is Down\n",
		   			at_driver_name,
                    netdev->name );
            adapter->link_speed = SPEED_0;
            netif_carrier_off(netdev);
            netif_stop_queue(netdev);
        }
    }
    schedule_work(&adapter->link_chg_task);
}
static inline void
at_clear_phy_int(struct at_adapter* adapter)
{
	uint16_t phy_data;
	
        spin_lock(&adapter->stats_lock);
	at_read_phy_reg(&adapter->hw, 19, &phy_data);
	spin_unlock(&adapter->stats_lock);
}

/**
 * at_intr - Interrupt Handler
 * @irq: interrupt number
 * @data: pointer to a network interface device structure
 * @pt_regs: CPU registers structure
 **/

static irqreturn_t
at_intr(int irq, void *data, struct pt_regs *regs)
{
    struct at_adapter *adapter = ((struct net_device*)data)->priv;
    uint32_t status;
    uint8_t update_rx;
    int max_ints = 10;
    
    if (0 == (status = adapter->cmb.cmb->int_stats))
        return IRQ_NONE;
        
    update_rx = 0;
    
    do {
        // clear CMB interrupt status at once
        adapter->cmb.cmb->int_stats = 0;
	
	if (status & ISR_GPHY) { // clear phy status
               at_clear_phy_int(adapter);
	}	       
        // clear ISR status, and Enable CMB DMA/Disable Interrupt
        AT_WRITE_REG(&adapter->hw, REG_ISR, status|ISR_DIS_INT);

        // check if SMB intr
        if (status & ISR_SMB)
        {
            at_inc_smb(adapter);
        }

        // check if PCIE PHY Link down
        if (status&ISR_PHY_LINKDOWN)
        {
            DEBUGOUT1("pcie phy linkdown %x", status);
            if(netif_running(adapter->netdev)) { // reset MAC
                AT_WRITE_REG(&adapter->hw, REG_IMR, 0);
                schedule_work(&adapter->pcie_dma_to_rst_task);
                return IRQ_HANDLED; 
            }
        }

        // check if DMA read/write error ?
        if (status&(ISR_DMAR_TO_RST|ISR_DMAW_TO_RST)) 
        {
            DEBUGOUT1("PCIE DMA RW error (status = 0x%x) !", status);
            //AT_WRITE_REG(&adapter->hw, REG_MASTER_CTRL, MASTER_CTRL_SOFT_RST);
            AT_WRITE_REG(&adapter->hw, REG_IMR, 0);
            schedule_work(&adapter->pcie_dma_to_rst_task);
            return IRQ_HANDLED; 
        }
 
        
        // link event
        if (status&ISR_GPHY)
        {
            adapter->soft_stats.tx_carrier_errors++;
            at_check_for_link(adapter);
        }

        // transmit event
        if (status&ISR_CMB_TX) {
            at_intr_tx(adapter);
        }
        
        // rx exception
        if (    status&(ISR_RXF_OV|
                        ISR_RFD_UNRUN|
                        ISR_RRD_OV|
                        ISR_HOST_RFD_UNRUN|
                        ISR_HOST_RRD_OV|
                        ISR_CMB_RX))    {
            if (status&(ISR_RXF_OV|
                        ISR_RFD_UNRUN|
                        ISR_RRD_OV|
                        ISR_HOST_RFD_UNRUN|
                        ISR_HOST_RRD_OV)) {
                DEBUGOUT1(KERN_INFO 
                          "Receive Exception ! status = 0x%x", status);
            }
            at_intr_rx(adapter);
        }


        if (--max_ints < 0)
            break;
        
    } while ((status = adapter->cmb.cmb->int_stats));
    
    // re-enable Interrupt
    AT_WRITE_REG(&adapter->hw, REG_ISR, ISR_DIS_SMB|ISR_DIS_DMA);
    return IRQ_HANDLED;
}

static void
at_intr_tx(struct at_adapter* adapter)
{
    struct at_tpd_ring *tpd_ring = &adapter->tpd_ring; 
    struct at_buffer* buffer_info;
    uint16_t sw_tpd_next_to_clean;
    uint16_t cmb_tpd_next_to_clean;
    uint8_t update = 0;


    sw_tpd_next_to_clean = (uint16_t)atomic_read(&tpd_ring->next_to_clean);
    cmb_tpd_next_to_clean = le16_to_cpu(adapter->cmb.cmb->tpd_cons_idx);

    while (cmb_tpd_next_to_clean != sw_tpd_next_to_clean) {
        tx_packet_desc_t* tpd;
        
        update = 1;
        
        tpd = AT_TPD_DESC(tpd_ring, sw_tpd_next_to_clean);
        buffer_info = &tpd_ring->buffer_info[sw_tpd_next_to_clean];
        if(buffer_info->dma) {
            pci_unmap_page(adapter->pdev,
                           buffer_info->dma,
                           buffer_info->length,
                           PCI_DMA_TODEVICE);
            buffer_info->dma = 0;
        }

        if(buffer_info->skb) {
            dev_kfree_skb_irq(buffer_info->skb);
            buffer_info->skb = NULL;
        }
        tpd->buffer_addr = 0;
        tpd->desc.data = 0;
        
        if (++sw_tpd_next_to_clean == tpd_ring->count)  
            sw_tpd_next_to_clean = 0;
    }
    atomic_set(&tpd_ring->next_to_clean,sw_tpd_next_to_clean);
    
    if(netif_queue_stopped(adapter->netdev) && 
       netif_carrier_ok(adapter->netdev))
        netif_wake_queue(adapter->netdev);
}

static inline void
at_rx_checksum(struct at_adapter* adapter, 
        rx_return_desc_t* rrd,
        struct sk_buff* skb)
{
    skb->ip_summed = CHECKSUM_NONE;

    if (rrd->pkt_flg&PACKET_FLAG_ERR) {
        if (rrd->err_flg&
            (ERR_FLAG_CRC|ERR_FLAG_TRUNC|ERR_FLAG_CODE|ERR_FLAG_OV)) {
            adapter->hw_csum_err++;
	    DEBUGOUT("checksum error !");
            return;
        }
    }

    // none IPv4 
    if (!(rrd->pkt_flg&PACKET_FLAG_IPV4)) {
        // checksum is invalid
	DEBUGOUT("checksum is invalid because of non-IPv4");
        return;
    }
    
    // IPv4 packet
    
    if (!(rrd->err_flg&(ERR_FLAG_IP_CHKSUM|ERR_FLAG_L4_CHKSUM)) ) {
		skb->ip_summed = CHECKSUM_UNNECESSARY;
		adapter->hw_csum_good++;
		DEBUGOUT("IPv4, checkesum ok");
		return;
    }
    
    // IPv4, but hardware think it's checksum is wrong
    DEBUGOUT1("hardware xsum wrong pkt_flag:%x, err_flag:%x", 
		    rrd->pkt_flg, rrd->err_flg);
    skb->ip_summed = CHECKSUM_HW;
    skb->csum = htons(rrd->xsz.xsum_sz.rx_chksum);
    adapter->hw_csum_err++;
    return;
}


static void
at_intr_rx(struct at_adapter* adapter)
{
    int i, count;
    uint16_t length, rrd_next_to_clean;
    struct at_rfd_ring *rfd_ring = &adapter->rfd_ring;
    struct at_rrd_ring *rrd_ring = &adapter->rrd_ring; 
    struct at_buffer * buffer_info;
    rx_return_desc_t* rrd;
    struct sk_buff *skb;
   
    count = 0; 
    
    rrd_next_to_clean = (uint16_t)atomic_read(&rrd_ring->next_to_clean);
    
    while (1)
    {
        rrd = AT_RRD_DESC(rrd_ring, rrd_next_to_clean);
        i = 1;
        if (rrd->xsz.valid) { // packet valid
chk_rrd:        
            // check rrd status
	    if (rrd->num_buf != 1) {
		DEBUGOUT1("RRD NumRfd %d", rrd->num_buf);
                DEBUGOUT1("packet length = %d", rrd->xsz.xsum_sz.pkt_size);
            } else {
		goto rrd_ok;
	    }
            
            // rrd seems to be bad
            if (i-- > 0) { // rrd may not be DMAed completely
		DEBUGOUT("RRD may not be DMAed completely");
                usec_delay(1);
                goto chk_rrd;
            }
            // bad rrd
            printk("BAD RRD\n");
	    // see if update RFD index
	    if (rrd->num_buf > 1) {
	        uint16_t num_buf;
	        num_buf = 
		(rrd->xsz.xsum_sz.pkt_size+adapter->rx_buffer_len - 1)/
			adapter->rx_buffer_len;
		DEBUGOUT1("RRD.buf_index (%d)", rrd->buf_indx);
		if (rrd->num_buf == num_buf) {
	            // clean alloc flag for bad rrd
                    while (rfd_ring->next_to_clean != 
		           (rrd->buf_indx + num_buf) ) {
			DEBUGOUT1("clear index (%d)", rfd_ring->next_to_clean);
	                rfd_ring->buffer_info[rfd_ring->next_to_clean].alloced = 0;
	                if (++rfd_ring->next_to_clean == 
			    rfd_ring->count) {
		            rfd_ring->next_to_clean = 0;
	                } 
	            } // end while 	    
		} // end if (rrd->num_buf == ...)
	    }

	    // update rrd
            rrd->xsz.valid = 0;
            if (++rrd_next_to_clean == rrd_ring->count)
                rrd_next_to_clean = 0;
            count++; 
            continue;
        } else { // current rrd still not be updated
          break;
        }

rrd_ok:
	//printk(KERN_INFO "Receving Packet!\n");
	// clean alloc flag for bad rrd
	while (rfd_ring->next_to_clean != rrd->buf_indx) {
	    rfd_ring->buffer_info[rfd_ring->next_to_clean].alloced = 0;
	    if (++rfd_ring->next_to_clean == rfd_ring->count) {
		rfd_ring->next_to_clean = 0;
	    }
	}	    
    
        buffer_info = &rfd_ring->buffer_info[rrd->buf_indx];
	if (++rfd_ring->next_to_clean == rfd_ring->count) {
	    rfd_ring->next_to_clean = 0;
	}
	

        // update rrd next to clean
        if (++rrd_next_to_clean == rrd_ring->count)
            rrd_next_to_clean = 0;
        count++; 
	
            
        if (rrd->pkt_flg&PACKET_FLAG_ERR) {
            if (!(rrd->err_flg & 
                (ERR_FLAG_IP_CHKSUM|ERR_FLAG_L4_CHKSUM|ERR_FLAG_LEN))) {
                /* packet error , don't need upstream */
                buffer_info->alloced = 0;  
		rrd->xsz.valid = 0;
                DEBUGOUT1("rrd error flag %x", rrd->err_flg);
                continue;
            }
        }
	
        /* Good Receive */
	
        pci_unmap_page(adapter->pdev,
                       buffer_info->dma,
                       buffer_info->length,
                       PCI_DMA_FROMDEVICE);
        skb = buffer_info->skb;
        length = le16_to_cpu(rrd->xsz.xsum_sz.pkt_size);
            
        skb_put(skb, length - ETHERNET_FCS_SIZE);

        /* Receive Checksum Offload */
        at_rx_checksum(adapter, rrd, skb); 
        skb->protocol = eth_type_trans(skb, adapter->netdev);

#ifdef NETIF_F_HW_VLAN_TX
        if(adapter->vlgrp && (rrd->pkt_flg&PACKET_FLAG_VLAN_INS)) {
            uint16_t vlan_tag = 
                    (rrd->vlan_tag>>4) |
                    ((rrd->vlan_tag&7) << 13) |
                    ((rrd->vlan_tag&8) << 9);
            DEBUGOUT1("RX VLAN TAG<RRD>=0x%04x", rrd->vlan_tag);
            vlan_hwaccel_rx(skb, adapter->vlgrp, vlan_tag);
        } else 
#endif
        netif_rx(skb);

	// let protocol layer free skb
        buffer_info->skb = NULL;  
        buffer_info->alloced = 0;  
        rrd->xsz.valid = 0;

        adapter->netdev->last_rx = jiffies;
    }
    
    atomic_set(&rrd_ring->next_to_clean, rrd_next_to_clean);
   
    at_alloc_rx_buffers(adapter);
 
    // update mailbox ?
    if (0 != count) {
        uint32_t tpd_next_to_use;
        uint32_t rfd_next_to_use;
        uint32_t rrd_next_to_clean;

        spin_lock(&adapter->mb_lock);
                
        tpd_next_to_use = atomic_read(&adapter->tpd_ring.next_to_use);
        rfd_next_to_use = (uint32_t)atomic_read(&adapter->rfd_ring.next_to_use);
        rrd_next_to_clean = (uint32_t)atomic_read(&adapter->rrd_ring.next_to_clean);
	//DEBUGOUT1("mailbox update, rfd_next_to_use (%d)", rfd_next_to_use);
        AT_WRITE_REG(&adapter->hw, REG_MAILBOX, 
                ((rfd_next_to_use & MB_RFD_PROD_INDX_MASK) << MB_RFD_PROD_INDX_SHIFT) |
                ((rrd_next_to_clean & MB_RRD_CONS_INDX_MASK) << MB_RRD_CONS_INDX_SHIFT) |
                ((tpd_next_to_use & MB_TPD_PROD_INDX_MASK) << MB_TPD_PROD_INDX_SHIFT) ) ;
                
        spin_unlock(&adapter->mb_lock);
    }
    
    
}


static inline uint16_t tpd_avail(struct at_tpd_ring *tpd_ring)
{
    uint16_t next_to_clean = (uint16_t)atomic_read(&tpd_ring->next_to_clean);
    uint16_t next_to_use = (uint16_t)atomic_read(&tpd_ring->next_to_use);
    return ((next_to_clean > next_to_use) ? 
            next_to_clean - next_to_use - 1 :
            tpd_ring->count + next_to_clean - next_to_use - 1);
}

static inline void
at_update_mailbox(struct at_adapter* adapter)
{
    unsigned long flags;
    uint32_t tpd_next_to_use;
    uint32_t rfd_next_to_use;
    uint32_t rrd_next_to_clean;
    
    spin_lock_irqsave(&adapter->mb_lock, flags);
    
    tpd_next_to_use = atomic_read(&adapter->tpd_ring.next_to_use);
    rfd_next_to_use = (uint32_t)atomic_read(&adapter->rfd_ring.next_to_use);
    rrd_next_to_clean = (uint32_t)atomic_read(&adapter->rrd_ring.next_to_clean);
    
    AT_WRITE_REG(&adapter->hw, REG_MAILBOX, 
        ((rfd_next_to_use & MB_RFD_PROD_INDX_MASK) << MB_RFD_PROD_INDX_SHIFT) |
        ((rrd_next_to_clean & MB_RRD_CONS_INDX_MASK) << MB_RRD_CONS_INDX_SHIFT) |
        ((tpd_next_to_use & MB_TPD_PROD_INDX_MASK) << MB_TPD_PROD_INDX_SHIFT) );
        
    spin_unlock_irqrestore(&adapter->mb_lock, flags);
}

static inline int
at_tso(struct at_adapter *adapter, struct sk_buff *skb, tso_param_t * tso)
{
#ifdef NETIF_F_TSO
    uint8_t ipofst;
    int err;
    
    if(skb_shinfo(skb)->tso_size) {
	    if(skb_header_cloned(skb)) {
	        err = pskb_expand_head(skb, 0, 0, GFP_ATOMIC);
	        if (err)
	            return err;
        }  

        if(skb->protocol == ntohs(ETH_P_IP)) {
            skb->nh.iph->tot_len = 0;
            skb->nh.iph->check = 0;
            skb->h.th->check = ~csum_tcpudp_magic(skb->nh.iph->saddr,
                                        skb->nh.iph->daddr,
                                        0,
                                        IPPROTO_TCP,
                                        0);
            ipofst = skb->nh.raw - skb->data;
            if (14 != ipofst)  // 802.3 frame
                tso->eth_type = 1;
        
            tso->iphl = skb->nh.iph->ihl;
            tso->tcp_hdrlen = skb->h.th->doff << 2;
            tso->mss = skb_shinfo(skb)->tso_size;
            tso->ip_chksum = 1;
            tso->tcp_chksum = 1;
            tso->segment = 1;
            return TRUE;
        }
    }
#endif

    return FALSE;
}

static inline int
at_tx_csum(struct at_adapter *adapter, struct sk_buff *skb, csum_param_t* csum)
{
    uint8_t css, cso;

    if(skb->ip_summed == CHECKSUM_HW) {
         
        cso = skb->h.raw - skb->data;
        css = (skb->h.raw + skb->csum) - skb->data;
        if(cso&0x1) {
            AT_ERR("payload offset != even number !\n");
            return  -1;
        }
        csum->payload_offset = cso;
        csum->xsum_offset = css;
        csum->custom_chksum = 1;
	//DEBUGOUT1("hardware chekcsum: css=%d, cso=%d", css, cso);
	

        return TRUE;
    }

    return FALSE;
}


static inline void
at_tx_map(struct at_adapter *adapter, struct sk_buff *skb, boolean_t tcp_seg)
{
    struct at_tpd_ring *tpd_ring = &adapter->tpd_ring;
    struct at_buffer *buffer_info;
    uint16_t first_buf_len = skb->len;
    struct page *page;                                                       
    unsigned long offset;  

    uint16_t tpd_next_to_use;
#ifdef NETIF_F_TSO
    uint16_t proto_hdr_len;
#endif

#ifdef MAX_SKB_FRAGS
    unsigned int nr_frags;
    unsigned int f;
    first_buf_len -= skb->data_len;
    nr_frags = skb_shinfo(skb)->nr_frags;
#endif//MAX_SKB_FRAGS

    
    tpd_next_to_use = (uint16_t) atomic_read(&tpd_ring->next_to_use);
    
    buffer_info = &tpd_ring->buffer_info[tpd_next_to_use]; 
    if (buffer_info->skb) {
	    BUG();
    }
    buffer_info->skb = NULL;    // put skb in last TPD

#ifdef NETIF_F_TSO
    if (tcp_seg) {
        proto_hdr_len = ((skb->h.raw - skb->data) + (skb->h.th->doff << 2));
        buffer_info->length = proto_hdr_len;
        page = virt_to_page(skb->data);                                            
        offset = (unsigned long)skb->data & ~PAGE_MASK;                           
        buffer_info->dma =
                        pci_map_page(adapter->pdev,
                                     page,
                                     offset,
                                     proto_hdr_len,
                                     PCI_DMA_TODEVICE);
                                     
        if(++tpd_next_to_use == tpd_ring->count) tpd_next_to_use = 0;
        
        if (first_buf_len > proto_hdr_len) {
        	uint16_t len12 = first_buf_len - proto_hdr_len;
        	uint16_t i, m = (len12 + MAX_TX_BUF_LEN - 1) / MAX_TX_BUF_LEN;
        	for (i=0; i < m; i++) {
        		buffer_info = &tpd_ring->buffer_info[tpd_next_to_use];
            	buffer_info->skb = NULL;
            	buffer_info->length = 
            		(MAX_TX_BUF_LEN >= len12) ? MAX_TX_BUF_LEN : len12;
            	len12 -= buffer_info->length;
                page = virt_to_page(skb->data+(proto_hdr_len+i*MAX_TX_BUF_LEN));                                            
            	offset = (unsigned long)(skb->data+(proto_hdr_len+i*MAX_TX_BUF_LEN)) & ~PAGE_MASK;
               	buffer_info->dma = 
                        pci_map_page(adapter->pdev,
                                     page, 
                                     offset,
                                     buffer_info->length,
                                     PCI_DMA_TODEVICE);
            	if(++tpd_next_to_use == tpd_ring->count) tpd_next_to_use = 0;  
            } 
        }
    } else {
#endif//NETIF_F_TSO
    
    buffer_info->length = first_buf_len;
    page = virt_to_page(skb->data);                                            
    offset = (unsigned long)skb->data & ~PAGE_MASK;
    buffer_info->dma =
                    pci_map_page(adapter->pdev,
                                 page,
                                 offset,
                                 first_buf_len,
                                 PCI_DMA_TODEVICE);
    if(++tpd_next_to_use == tpd_ring->count) tpd_next_to_use = 0;   
#ifdef NETIF_F_TSO
    }    
#endif//NETIF_F_TSO

#ifdef MAX_SKB_FRAGS
    for(f = 0; f < nr_frags; f++) {
        struct skb_frag_struct *frag;
		uint16_t lenf, i, m;
        
        frag = &skb_shinfo(skb)->frags[f];
        lenf = frag->size;
        
		m = (lenf + MAX_TX_BUF_LEN - 1) / MAX_TX_BUF_LEN;
        for (i=0; i < m; i++) {
        	buffer_info = &tpd_ring->buffer_info[tpd_next_to_use];
        	if (buffer_info->skb) {
		    	BUG();
	    	}
            buffer_info->skb = NULL;
            buffer_info->length = 
            	(lenf > MAX_TX_BUF_LEN) ? MAX_TX_BUF_LEN : lenf;
            lenf -= buffer_info->length;
            
			buffer_info->dma =
                pci_map_page(adapter->pdev,
                             frag->page,
                             frag->page_offset+i*MAX_TX_BUF_LEN,
                             buffer_info->length,
                             PCI_DMA_TODEVICE);
                             
            if(++tpd_next_to_use == tpd_ring->count) tpd_next_to_use = 0; 
        } 
    }
#endif//MAX_SKB_FRAGS
    
    // last tpd's buffer-info
    buffer_info->skb = skb;
}

static inline void
at_tx_queue(struct at_adapter *adapter, int count, tpd_descr_t * descr)
{
    struct at_tpd_ring *tpd_ring = &adapter->tpd_ring;
    int j;
    struct at_buffer * buffer_info;
    tx_packet_desc_t* tpd;
    uint16_t tpd_next_to_use = (uint16_t)atomic_read(&tpd_ring->next_to_use);
    
   // DEBUGOUT1("<at_tx_quue> count=%d, tpd_desc=%llx", count, descr->data);
    
    for(j=0; j<count; j++) {
        buffer_info = &tpd_ring->buffer_info[tpd_next_to_use];
        tpd = AT_TPD_DESC(&adapter->tpd_ring, tpd_next_to_use);
        tpd->buffer_addr = cpu_to_le64(buffer_info->dma);
        tpd->desc.data = descr->data;
        tpd->desc.csum.buf_len = cpu_to_le16(buffer_info->length);
       /* 
	DEBUGOUT1("<at_tx_quue> TPD_NEXT_TO_USE: %d buf_len=%d", 
			tpd_next_to_use,
			buffer_info->length);
      	*/
#ifdef NETIF_F_TSO
        if ((descr->tso.segment)&& (0==j)) 
            tpd->desc.tso.hdr_flg = 1;
#endif//NETIF_F_TSO

        if (j==(count-1))
            tpd->desc.csum.eop = 1;
       
        if(++tpd_next_to_use == tpd_ring->count) tpd_next_to_use = 0;
    }

    /* Force memory writes to complete before letting h/w
     * know there are new descriptors to fetch.  (Only
     * applicable for weak-ordered memory model archs,
     * such as IA-64). */
    wmb();
    
    atomic_set(&tpd_ring->next_to_use, (int)tpd_next_to_use);
}


static int
at_xmit_frame(struct sk_buff *skb, struct net_device *netdev)
{
    struct at_adapter *adapter = netdev_priv(netdev);
    unsigned int len = skb->len;
    unsigned long flags;
    unsigned int nr_frags = 0;
    int tso;
    int count = 1;
    tpd_descr_t param;
    
#ifdef NETIF_F_TSO
    unsigned int mss = 0;
#endif
#ifdef MAX_SKB_FRAGS
	unsigned int f;
	len -= skb->data_len;
#endif

    //DEBUGFUNC("at_xmit_frame");

    if(skb->len <= 0) {
        dev_kfree_skb_any(skb);
        return NETDEV_TX_OK;
    }
    
    param.data = 0;
    
#ifdef MAX_SKB_FRAGS
    nr_frags = skb_shinfo(skb)->nr_frags;
    for(f = 0; f < nr_frags; f++) {
    	uint16_t fg_size = skb_shinfo(skb)->frags[f].size;
        if (fg_size) {
        	count += (fg_size + MAX_TX_BUF_LEN - 1) / MAX_TX_BUF_LEN;
        }
    }
#endif//MAX_SKB_FRAGS

#ifdef NETIF_F_TSO
    mss = skb_shinfo(skb)->tso_size;

    // first buffer must be large(or equal) than protocol header length. 
    if(mss) {
        unsigned int proto_hdr_len;
        if(skb->protocol == ntohs(ETH_P_IP)) {
            proto_hdr_len = ((skb->h.raw - skb->data) + (skb->h.th->doff << 2));
            if (proto_hdr_len > len) {
                printk(KERN_INFO "protocol header length = 0x%x, buffer header length = 0x%x\n", 
                    proto_hdr_len, len);
                dev_kfree_skb_any(skb);
		        return NETDEV_TX_OK;
            }
            // need additional TPD ?
            if (proto_hdr_len != len) {
            	count += (len - proto_hdr_len + MAX_TX_BUF_LEN - 1) / MAX_TX_BUF_LEN;
            }
        }
    }
#endif//NETIF_F_TSO
#ifdef NETIF_F_LLTX
	local_irq_save(flags);
	if(!spin_trylock(&adapter->tx_lock)) {
		/* Collision - tell upper layer to requeue */
		local_irq_restore(flags);
		return NETDEV_TX_LOCKED;
	}
#else
	spin_lock_irqsave(&adapter->tx_lock, flags);
#endif
    if(tpd_avail(&adapter->tpd_ring) < count) {
        // no enough descriptor
        netif_stop_queue(netdev);
		spin_unlock_irqrestore(&adapter->tx_lock, flags);
		return NETDEV_TX_BUSY;
    }

#ifndef NETIF_F_LLTX
	spin_unlock_irqrestore(&adapter->tx_lock, flags);
#endif

    param.data = 0;
    
#ifdef NETIF_F_HW_VLAN_TX
	if(adapter->vlgrp && vlan_tx_tag_present(skb)) {
            uint16_t vlan_tag = vlan_tx_tag_get(skb);
            vlan_tag = (vlan_tag << 4)          |
                       (vlan_tag >> 13)         |
                       ((vlan_tag >>9) & 0x8)   ;
	    param.csum.ins_vlag = 1;
	    param.csum.valan_tag = vlan_tag;
            DEBUGOUT1("TX VLAN TAG<TPD>=%04x", vlan_tag);
	}
#endif    
    
    tso = at_tso(adapter, skb, &param.tso);
    if (tso < 0) {
#ifdef NETIF_F_LLTX
       spin_unlock_irqrestore(&adapter->tx_lock, flags);
#endif
       dev_kfree_skb_any(skb);
	    return NETDEV_TX_OK; 
	}  
	
	if (!tso) {
	    if (at_tx_csum(adapter, skb, &param.csum) < 0) {
	        spin_unlock_irqrestore(&adapter->tx_lock, flags);
            dev_kfree_skb_any(skb);
	        return NETDEV_TX_OK;
	    }
    }
    
    at_tx_map(adapter, skb, 1==param.csum.segment);
    at_tx_queue(adapter, count, &param);
    
    netdev->trans_start = jiffies;
    
#ifdef NETIF_F_LLTX
	spin_unlock_irqrestore(&adapter->tx_lock, flags);
#endif
    
    // update mailbox
    at_update_mailbox(adapter);
    
    
    return NETDEV_TX_OK;
}




static void at_inc_smb(struct at_adapter * adapter)
{
    stats_msg_block_t* smb = adapter->smb.smb;
    
    /* Fill out the OS statistics structure */
    adapter->soft_stats.rx_packets          += smb->rx_ok;
    adapter->soft_stats.tx_packets          += smb->tx_ok;
    adapter->soft_stats.rx_bytes            += smb->rx_byte_cnt;
    adapter->soft_stats.tx_bytes            += smb->tx_byte_cnt;
    adapter->soft_stats.multicast           += smb->rx_mcast;
    adapter->soft_stats.collisions          += (smb->tx_1_col + 
                                                smb->tx_2_col*2 + 
                                                smb->tx_late_col + 
                                                smb->tx_abort_col * adapter->hw.max_retry);

    /* Rx Errors */

    adapter->soft_stats.rx_errors           += (smb->rx_frag +
                                                smb->rx_fcs_err + 
                                                smb->rx_len_err + 
                                                smb->rx_sz_ov + 
                                                smb->rx_rxf_ov + 
                                                smb->rx_rrd_ov +
                                                smb->rx_align_err );
    adapter->soft_stats.rx_fifo_errors      += smb->rx_rxf_ov;
    adapter->soft_stats.rx_length_errors    += smb->rx_len_err;
    adapter->soft_stats.rx_crc_errors       += smb->rx_fcs_err;
    adapter->soft_stats.rx_frame_errors     += smb->rx_align_err;
    adapter->soft_stats.rx_missed_errors    += (smb->rx_rrd_ov + 
                                                smb->rx_rxf_ov);
                                                
    adapter->soft_stats.rx_pause            += smb->rx_pause;
    adapter->soft_stats.rx_rrd_ov           += smb->rx_rrd_ov; 
    adapter->soft_stats.rx_trunc            += smb->rx_sz_ov;                                               

    /* Tx Errors */

    adapter->soft_stats.tx_errors           += (smb->tx_late_col +
                                                smb->tx_abort_col +
                                                smb->tx_underrun + 
                                                smb->tx_trunc );
    adapter->soft_stats.tx_fifo_errors      += smb->tx_underrun;
    adapter->soft_stats.tx_aborted_errors   += smb->tx_abort_col;
    adapter->soft_stats.tx_window_errors    += smb->tx_late_col;
    
    adapter->soft_stats.excecol             += smb->tx_abort_col;
    adapter->soft_stats.deffer              += smb->tx_defer;
    adapter->soft_stats.scc                 += smb->tx_1_col;
    adapter->soft_stats.mcc                 += smb->tx_2_col;
    adapter->soft_stats.latecol             += smb->tx_late_col;
    adapter->soft_stats.tx_underun          += smb->tx_underrun;
    adapter->soft_stats.tx_trunc            += smb->tx_trunc;
    adapter->soft_stats.tx_pause            += smb->tx_pause;


    // finally
        
    adapter->net_stats.rx_packets           = adapter->soft_stats.rx_packets;                  
    adapter->net_stats.tx_packets           = adapter->soft_stats.tx_packets;                     
    adapter->net_stats.rx_bytes             = adapter->soft_stats.rx_bytes;                     
    adapter->net_stats.tx_bytes             = adapter->soft_stats.tx_bytes;            
    adapter->net_stats.multicast            = adapter->soft_stats.multicast;           
    adapter->net_stats.collisions           = adapter->soft_stats.collisions;          
    adapter->net_stats.rx_errors            = adapter->soft_stats.rx_errors;          
    adapter->net_stats.rx_over_errors       = adapter->soft_stats.rx_missed_errors;
    adapter->net_stats.rx_length_errors     = adapter->soft_stats.rx_length_errors;    
    adapter->net_stats.rx_crc_errors        = adapter->soft_stats.rx_crc_errors;       
    adapter->net_stats.rx_frame_errors      = adapter->soft_stats.rx_frame_errors;     
    adapter->net_stats.rx_fifo_errors       = adapter->soft_stats.rx_fifo_errors;      
    adapter->net_stats.rx_missed_errors     = adapter->soft_stats.rx_missed_errors;    
    adapter->net_stats.tx_errors            = adapter->soft_stats.tx_errors;           
    adapter->net_stats.tx_fifo_errors       = adapter->soft_stats.tx_fifo_errors;      
    adapter->net_stats.tx_aborted_errors    = adapter->soft_stats.tx_aborted_errors;  
    adapter->net_stats.tx_window_errors     = adapter->soft_stats.tx_window_errors;    
    adapter->net_stats.tx_carrier_errors    = adapter->soft_stats.tx_carrier_errors;  
}



/**
 * at_phy_config - Timer Call-back
 * @data: pointer to netdev cast into an unsigned long
 **/

static void
at_phy_config(unsigned long data)
{
    struct at_adapter *adapter = (struct at_adapter *) data;
    struct at_hw *hw = &adapter->hw;  
    unsigned long flags;

    DEBUGFUNC("at_phy_reconfig!");
    
    spin_lock_irqsave(&adapter->stats_lock, flags);
    adapter->phy_timer_pending = FALSE;
    at_write_phy_reg(hw, MII_ADVERTISE, hw->mii_autoneg_adv_reg);
    at_write_phy_reg(hw, MII_AT001_CR, hw->mii_1000t_ctrl_reg);
    DEBUGOUT("4&9 registers written");
    at_write_phy_reg(hw, MII_BMCR, MII_CR_RESET|MII_CR_AUTO_NEG_EN);
    spin_unlock_irqrestore(&adapter->stats_lock, flags);
}


/**
 * at_watchdog - Timer Call-back
 * @data: pointer to netdev cast into an unsigned long
 **/

static void
at_watchdog(unsigned long data)
{
    struct at_adapter *adapter = (struct at_adapter *) data;
 //   struct at_hw *hw = &adapter->hw;
 //   uint16_t phy_data;

   // DEBUGFUNC("at_watchdog!");

    // update LinkStatus
//    at_read_phy_reg(hw, MII_BMSR, &phy_data);

/*
    if(netif_queue_stopped(adapter->netdev) && 
       netif_carrier_ok(adapter->netdev)) {
	    
        //netif_wake_queue(adapter->netdev);
	DEBUGFUNC("queue stopped !\n");
    }
*/
    /* Reset the timer */
    mod_timer(&adapter->watchdog_timer, jiffies + 2 * HZ);
}


/**
 * When ACPI resume on some VIA MotherBoard, the Interrupt Disable bit/0x400
 * on PCI Command register is disable.
 * The function enable this bit.
 * Brackett, 2006/03/15
 */
static void at_via_workaround(struct at_adapter * adapter)
{
	unsigned long value;
	value = AT_READ_REGW(&adapter->hw, PCI_COMMAND);
	if (value & PCI_COMMAND_INTX_DISABLE)
	{
		value &= ~PCI_COMMAND_INTX_DISABLE;
	}
	
	AT_WRITE_REG(&adapter->hw, PCI_COMMAND, value);
}
